import os
import sys
import argparse
import json
import logging
import logging.config
from ast import literal_eval
from importlib.util import spec_from_file_location, module_from_spec

from renki.common.utilities.module_loader import import_authentication_modules, import_modules

default_logging_config = {
    'version': 1,
    'disable_existing_loggers': True,
    'formatters': {
        'verbose': {
            'format': '%(levelname)s %(asctime)s %(module)s %(process)d %(thread)d %(message)s'
        },
        'simple': {
            '()': 'logging.Formatter',
            'format': '%(asctime)-20s %(levelname)s %(module)s %(message)s'
        }
    },
    'filters': {
    },
    'handlers': {
        'null': {
            'level': 'DEBUG',
            'class': 'logging.NullHandler',
        },
        'console': {
            'class': 'logging.StreamHandler',
            'formatter': 'verbose'
        }
    },
    'loggers': {
        'server': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'admin': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'INFO',
        },
        'utils': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'ticket': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'INFO',
        },
        'module_database': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'module_dns_zone': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'module_domain': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'module_port': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'module_repository': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'module_dummy': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'tickets_done': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'RenkiSocket': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
    }
}


class SettingsError(Exception):
    def __init__(self, option, msg):
        err_msg = 'Settings validation failed on {option}. Reason: {msg}'.format(option=option, msg=msg)
        super(SettingsError, self).__init__(err_msg)


class SettingsParser(argparse.ArgumentParser):
    def __init__(self, *args, **kwargs):
        self.defaults = {}
        self.env_keys = []
        self.requirements = {}

        super(SettingsParser, self).__init__(*args, **kwargs)

        # Add our default config argument
        self.add_argument('-c', '--config', action='store', help='Location of your configuration file')

    def add_argument(self, *args, **kwargs):
        # Capture requirements for separated validation after source merge
        requirements = {}
        requirement_keys = [
            'type',
            'required'
        ]
        for key in requirement_keys:
            if key in kwargs:
                requirements[key] = kwargs[key]
                del kwargs[key]

        # Add argument
        action = super(SettingsParser, self).add_argument(*args, **kwargs)

        # Store actions that have defaults so we can prevent overwrites in validation phase
        if 'default' in kwargs:
            self.defaults[action.dest] = kwargs['default']

        # Append argument action to be configurable from the environment
        self.env_keys.append(action.dest.upper())

        # Store requirements for final validation
        self.requirements[action.dest.upper()] = requirements

    def validate_value_type(self, option_name: str, value: any, target_type: type):
        type_func = super(SettingsParser, self)._registry_get('type', target_type, target_type)

        if not callable(type_func):
            msg = 'Type validator {func} is not callable'.format(func=type_func)
            raise SettingsError(option_name, msg)

        try:
            result = type_func(value)
        except argparse.ArgumentTypeError:
            name = getattr(target_type, '__name__', repr(target_type))
            msg = str(sys.exc_info()[1])
            raise SettingsError(option_name, '{name}: {msg}'.format(name=name, msg=msg))
        except (TypeError, ValueError):
            msg = 'Invalid {type} value: {value}'.format(type=getattr(target_type, '__name__', repr(target_type)),
                                                         value=value)
            raise SettingsError(option_name, msg)

        return result

    def parse_known_args(self, args=None, namespace=None):
        parsed_namespace, parsed_args = super(SettingsParser, self).parse_known_args(args=args, namespace=namespace)
        final_namespace = argparse.Namespace()

        # Load base from given configuration file
        if parsed_namespace.config:
            if not os.path.isfile(parsed_namespace.config):
                raise FileNotFoundError(parsed_namespace.config)

            spec = spec_from_file_location('parsed_settings', parsed_namespace.config)
            settings = module_from_spec(spec)
            spec.loader.exec_module(settings)
            for key, value in settings.__dict__.items():
                if not key.startswith('__') and not callable(key):
                    setattr(final_namespace, key, value)

        # Overwrite from environment variables
        for key in self.env_keys:
            if key in os.environ:
                try:
                    setattr(final_namespace, key, literal_eval(os.environ[key]))
                except (ValueError, SyntaxError):
                    setattr(final_namespace, key, str(os.environ[key]))

        # Write parsed arguments into namespace
        for key, value in parsed_namespace.__dict__.items():
            if not key.startswith('__') and not callable(key):
                if not hasattr(final_namespace, key.upper()) or (value is not None and value != self.defaults[key]):
                    setattr(final_namespace, key.upper(), value)

        # Validate final configuration
        for key in self.requirements.keys():
            requirements = self.requirements[key]
            if 'required' in requirements and requirements['required'] and \
                    (not hasattr(final_namespace, key) or getattr(final_namespace, key) is None):
                raise SettingsError(key, key + ' is a required option. See --help or manual for more information.')
            if 'type' in requirements and requirements['type'] is not None:
                required_type = requirements['type']
                value = getattr(final_namespace, key)
                self.validate_value_type(key, value, required_type)

        return final_namespace, parsed_args


class SettingsContainer:
    def from_dict(self, value_dict):
        for key, value in value_dict.items():
            if not key.startswith('__') and not callable(key):
                self.set(key, value)

    def from_args(self, namespace: argparse.Namespace):
        self.from_dict(namespace.__dict__)

    def _get_kwargs(self):
        return sorted(self.__dict__.items())

    def __repr__(self):
        type_name = type(self).__name__
        value_strings = []
        for name, value in self._get_kwargs():
            if name == 'parser':
                continue
            value_strings.append('%s=%r' % (name, value))
        return '%s(%s)' % (type_name, ', '.join(value_strings))

    def set(self, key: str, value: any):
        setattr(self, key, value)


class Settings(SettingsContainer):
    # TODO: Handle unknown arguments
    def __init__(self, *options):
        self.options = options
        self.parser = SettingsParser(add_help=False)
        self.parser.add_argument('-?', '--help', action='help', help='Show this help message and exit.')
        self.parser.add_argument('--logging-config', type=str, default=None)

        for option in options:
            self.parser.add_argument(*option[0], **option[1])

    def _initialize_modules(self, context):
        # Import authentication module
        if context is 'core':
            self.AUTHENTICATION_MODULES = import_authentication_modules(self.AUTHENTICATION_MODULES)
        self.MODULES = import_modules(self.MODULES, context)

    def _configure_logging(self):
        if hasattr(self, 'LOGGING_CONFIG'):
            with open(self.LOGGING_CONFIG) as json_data:
                config = json.load(json_data)
                logging.config.dictConfig(config)
        else:
            logging.config.dictConfig(default_logging_config)

    def from_static(self, context: str, value_dict: dict):
        self.from_dict(value_dict)
        self._initialize_modules(context)
        self._configure_logging()

    def from_cli(self, context: str):
        try:
            parsed_args = self.parser.parse_args()
        except SettingsError as error:
            print(error)
            sys.exit(-1)
        super(Settings, self).from_args(parsed_args)
        self._initialize_modules(context)
        self._configure_logging()
