import argparse
import os

from .value_container import ValueContainer
from .schema import SchemaField, Schema


class Configuration(ValueContainer):
    def __init__(self, *schema: SchemaField, env_var_prefix=None):
        """
        Initializes a new ConfParser instance.
        :param schema: *SchemaField

        :Example:

        >>> from renki.common.conf import Configuration
        >>> parser = Configuration(
        >>>     SchemaField('debug',
        >>>                 flags=['-d', '--debug'],
        >>>                 action='store_true',
        >>>                 default=False,
        >>>                 help='Switch to enable debugging mode',),
        >>>     SchemaField('name',
        >>>                 flags=['-n', '--name'],
        >>>                 fromEnv=False,
        >>>                 type=str,
        >>>                 required=True,
        >>>                 help='Name of the user',),
        >>> )
        """
        super().__init__()
        self.parsed = False
        self.schema = Schema(*schema)

        self.env_var_prefix = env_var_prefix

        self.parser = argparse.ArgumentParser()
        for arg in self.schema.build_args():
            self.parser.add_argument(*arg[0], **arg[1])

    def add_field(self, name, **kwargs):
        if self.parsed:
            raise RuntimeError('Configuration has already been parsed')

        field = SchemaField(name, **kwargs)
        self.schema.fields[field.name] = field

        fields, props = field.build_arg()
        if fields is not None:
            self.parser.add_argument(fields, props)

    def parse(self, *value_dicts: dict, args=True, env_vars=True):
        if self.parsed:
            raise RuntimeError('Configuration has already been parsed')

        for value_dict in value_dicts:
            self.merge_from_dict(value_dict)

        if env_vars:
            self._parse_env_vars()

        if args:
            self._parse_args(args)

        self.schema.validate_values(self)

    def _parse_env_vars(self):
        for key, field in self.schema.fields.items():
            if field.from_env:
                env_key = field.env_key(self.env_var_prefix)
                if env_key in os.environ:
                    raw_val = os.environ[env_key]
                    if field.action is 'store_true':
                        self[key] = True
                    elif field.action is 'store_false':
                        self[key] = False
                    else:
                        value = raw_val
                        if field.type is list:
                            value = value.split(',')
                        value = field.type(value)
                        errors = field.validate_value(value)
                        if errors is not None:
                            raise ValueError(errors)
                        self[key] = value

    def _parse_args(self, args):
        if isinstance(args, list):
            parsed_args = self.parser.parse_args(args)
        else:
            parsed_args = self.parser.parse_args()
        for key, value in vars(parsed_args).items():
            self[key] = value
