import os
import unittest

from renki.common.conf import Configuration, SchemaField


class TestConfParser(unittest.TestCase):
    def setUp(self):
        self.confParser = Configuration(
            SchemaField('debug',
                        flags=['-d', '--debug'],
                        default=False,
                        action='store_true',
                        help='Switch to enable debugging mode',),
            SchemaField('name',
                        flags=['-n', '--name'],
                        from_env=False,
                        type=str,
                        required=True,
                        help='Name of the user',),
            SchemaField('count',
                        flags=['-c', '--count'],
                        type=int,
                        required=True,
                        help='Name of the user',),
            SchemaField('my_list',
                        flags=['-l', '--my-list'],
                        type=list,
                        help='My list of stuff'),
            env_var_prefix='RECONF'
        )

    def test_should_parse_from_dict(self):
        self.confParser.parse({
            'debug': True,
            'name': 'Test User'
        }, args=False, env_vars=False)
        self.assertTrue(self.confParser.debug)
        self.assertEqual('Test User', self.confParser.name)

    def test_should_parse_from_env_vars(self):
        os.environ['RECONF_DEBUG'] = 'True'
        os.environ['RECONF_NAME'] = 'Test User'
        os.environ['RECONF_COUNT'] = '10'
        os.environ['RECONF_MY_LIST'] = 'one,two,tree'
        self.confParser.parse(args=False)
        self.assertTrue('debug' in self.confParser, 'Expected \'debug\' to be defined on parsed config')
        self.assertTrue(self.confParser.debug, 'Expected \'debug\' to be true')
        self.assertFalse('name' in self.confParser, 'Expected \'name\' not to be parsed from env')
        self.assertTrue('count' in self.confParser, 'Expected \'count\' to be defined on parsed config')
        self.assertEqual(10, self.confParser.count, 'Expected \'count\' to equal to target value')
        self.assertTrue('my_list' in self.confParser, 'Expected \'my_list\' to be defined on parsed config')
        self.assertEqual(3, len(self.confParser.my_list), 'Expected \'my_list\' to contain three elements')

    def test_should_parse_from_args(self):
        args = ['-d', '-n', 'Test User', '--count', '10']
        self.confParser.parse(args=args)
        self.assertTrue('debug' in self.confParser, 'Expected \'debug\' to be defined on parsed config')
        self.assertTrue('name' in self.confParser, 'Expected \'name\' to be defined on parsed config')
        self.assertTrue('count' in self.confParser, 'Expected \'count\' to be defined on parsed config')
