import unittest

from renki.common.conf import ValueContainer, SchemaField, Schema


class TestSchemaField(unittest.TestCase):
    def setUp(self):
        self.testField = SchemaField('debug',
                                     flags=['-d', '--debug'],
                                     type=bool,
                                     default=False,
                                     help='Switch to enable debugging mode',)

    def test_should_give_an_argparseable_tuple(self):
        flags, props = self.testField.build_arg()
        self.assertIsInstance(flags, list)
        self.assertIsInstance(props, dict)

    def test_should_give_a_valid_env_key(self):
        key = self.testField.env_key()
        self.assertEqual('DEBUG', key)

    def test_should_give_a_valid_prefixed_env_key(self):
        key = self.testField.env_key('RECONF')
        self.assertEqual('RECONF_DEBUG', key)

    def test_validate_valid_value(self):
        self.assertEqual(self.testField.validate_value(True), None)

    def test_validate_invalid_value(self):
        error = self.testField.validate_value('test')
        self.assertTrue('debug has an invalid type' in error)


class TestSchema(unittest.TestCase):
    def setUp(self):
        self.testValues = ValueContainer()
        self.testValues.merge_from_dict({
            'debug': True,
            'name': 'Test User'
        })
        self.testValuesWithUndefined = self.testValues.copy()
        self.testValuesWithUndefined.extra = 'extra'

        self.testSchema = Schema(
            SchemaField('debug',
                        flags=['-d', '--debug'],
                        type=bool,
                        default=False,
                        help='Switch to enable debugging mode', ),
            SchemaField('name',
                        type=str,
                        help='Name for the user', )
        )

    def test_should_validate_values_correctly(self):
        self.testSchema.validate_values(self.testValues)

    def test_should_give_an_argparseable_list_of_args(self):
        args = self.testSchema.build_args()
        self.assertIsInstance(args, list, 'Expected args to be a list of args')
        self.assertEqual(1, len(args), 'Expected schema to build a single arg')
        self.assertIsInstance(args[0][0], list, 'Expected the first field of an arg to be an array')
        self.assertIsInstance(args[0][1], dict, 'Expected the second field of an arg to be a dict')
