import argparse
import json
import os

from .value_container import ValueContainer
from .schema import SchemaField, Schema


class Configuration(ValueContainer):
    def __init__(self, *schema: SchemaField, env_var_prefix=None):
        """
        Initializes a new ConfParser instance.
        :param schema: *SchemaField

        :Example:

        >>> from renki.common.conf import Configuration
        >>> parser = Configuration(
        >>>     SchemaField('debug',
        >>>                 flags=['-d', '--debug'],
        >>>                 action='store_true',
        >>>                 default=False,
        >>>                 help='Switch to enable debugging mode',),
        >>>     SchemaField('name',
        >>>                 flags=['-n', '--name'],
        >>>                 fromEnv=False,
        >>>                 type=str,
        >>>                 required=True,
        >>>                 help='Name of the user',),
        >>> )
        """
        super().__init__()
        self._parsed = False
        self._schema = Schema(*schema)

        self._env_var_prefix = env_var_prefix

        self._parser = argparse.ArgumentParser()
        for arg in self._schema.build_args():
            self._parser.add_argument(*arg[0], **arg[1])

        self.add_field('config', flags=['-c', '--config'], type=str, help='Path to a JSON configuration file')

    def add_field(self, name, **kwargs):
        if self._parsed:
            raise RuntimeError('Configuration has already been parsed')

        field = SchemaField(name, **kwargs)
        self._schema.fields[field.name] = field

        fields, props = field.build_arg()
        if fields is not None:
            self._parser.add_argument(*fields, **props)

    def parse(self, *value_dicts: dict, args=True, env_vars=True):
        if self._parsed:
            raise RuntimeError('Configuration has already been parsed')

        for value_dict in value_dicts:
            self.merge_from_dict(value_dict)

        if env_vars:
            self._parse_env_vars()

        if args:
            self._parse_args(args)

        if hasattr(self, 'config') and self.config is not None:
            self._parse_json_file(self.config)

        self._schema.validate_values(self)
        self._parsed = True

    def _parse_env_vars(self):
        for key, field in self._schema.fields.items():
            if field.from_env:
                env_key = field.env_key(self._env_var_prefix)
                if env_key in os.environ:
                    raw_val = os.environ[env_key]
                    if field.action is 'store_true':
                        self[key] = True
                    elif field.action is 'store_false':
                        self[key] = False
                    else:
                        value = raw_val
                        if field.type is list:
                            value = value.split(',')
                        value = field.type(value)
                        errors = field.validate_value(value)
                        if errors is not None:
                            raise ValueError(errors)
                        self[key] = value
                elif field.default is not None:
                    self[key] = field.default

    def _parse_args(self, args):
        if isinstance(args, list):
            parsed_args = self._parser.parse_args(args)
        else:
            parsed_args = self._parser.parse_args()
        self.merge_from_dict(vars(parsed_args))

    def _parse_json_file(self, file_path: str):
        with open(file_path) as file:
            data = json.load(file)
            self.merge_from_dict(data)

    def __repr__(self):
        type_name = type(self).__name__
        value_strings = []
        for key, value in self._get_kwargs():
            if key.startswith('_'):
                continue
            value_strings.append('%s=%r' % (key, value))
        return '%s(%s)' % (type_name, ', '.join(value_strings))
