import logging
from json import dumps

import requests

from ..exceptions import *

logger = logging.getLogger("RenkiClient")


class RenkiClient(object):
    """
    Client library for Renki service management system
    >>> r = RenkiClient('http://localhost:8080/')
    >>> r.authenticate('test', 'test')
    >>> r.get('/domains')
    {'domains': [
           {'user_id': 1, 'id': 1, 'name': 'example.com'}
        ],
     'status': 'OK'
    }
    """
    def __init__(self, address):
        """
        @param address: address to Renki server api
        @type address: string
        """
        self.address = address.rstrip('/')
        self._session = requests.Session()
        self._session.headers = {'User-Agent': 'RenkiClient v0.1'}

    def set_token(self, token):
        """
        Set authentication token
        """
        self._session.headers['Authorization'] = token

    def authenticate(self, username, password):
        """
        Authenticate to server
        @param username: User username
        @type username: string
        @param password: User password
        @type password: string
        """
        credentials = {'username': username, 'password': password}
        ret = {}
        try:
            ret = self.post('/login', credentials)
        except NotAuthenticated:
            pass
        except ConnectionRefused:
            raise AuthenticationFailed('Failed to connect to Renki core')
        except HTTPException:
            raise AuthenticationFailed('Cannot authenticate due to server error')
        except Exception:
            print('fuu', Exception)
            raise
        if 'authToken' not in ret:
            raise AuthenticationFailed('Username or password invalid')
        self.set_token(ret['authToken'])
    auth = authenticate

    def _process(self, res):
        """
        Raise exception if code isn't 200
        """
        code = int(res.status_code)
        error = None
        try:
            _json = res.json()
            status = _json['status'] if 'status' in _json else 'OK'
        except (ValueError or KeyError):
            raise InvalidResponse("Got invalid response from server")
        if code == 200 and status == 'OK':
            return _json
        try:
            error = _json['error']
        except KeyError:
            raise InvalidResponse("Got invalid response from server")
        info = None
        try:
            info = _json['info']
        except KeyError:
            pass
        if code == 400 or status == 'ERROR':
            raise InvalidRequest(error, info=info)
        elif code == 401 or status == 'NOT_AUTHENTICATED':
            raise NotAuthenticated(error, info=info)
        elif code == 403 or status == 'DENIED':
            raise NotAuthorized(error, info=info)
        elif code == 404 or status == 'NOT_FOUND':
            raise NotFound(error, info=info)
        elif code == 405 or status == 'NOTALLOWD':
            raise MethodNotAllowed(error, info=info)
        elif code == 409 or status == 'CONFLICT':
            raise Conflict(error, info=info)
        elif code == 500 or status == 'SERVER_ERROR':
            raise ServerError(error, info=info)
        else:
            raise RenkiException(error, info=info)

    def _abs_url(self, path):
        """
        Return absolute url
        """
        return '/'.join([self.address, path.lstrip('/')])

    def wrap(self, cmd, *args, **kwargs):
        try:
            return cmd(*args, **kwargs)
        except requests.ConnectionError as e:
            raise ConnectionRefused("Cannot connect to server") from e
        except Exception as e:
            raise RenkiException("Cannot make query to server") from e

    def get(self, path, params=None):
        """
        @param path: API path, eg. domain
        @type path: string
        @param params: optional params for query
        @type params: dict
        """
        if params is None:
            params = {}

        ret = self.wrap(self._session.get, self._abs_url(path), params=params)
        return self._process(ret)

    def post(self, path, params=None, json=None):
        """
        @param path: API path, eg. domain
        @type path: string
        @param params: optional params for query
        @type params: dict
        """
        if params is None:
            params = {}

        ret = self.wrap(self._session.post, self._abs_url(path), data=params, json=json)
        return self._process(ret)

    def put(self, path, params=None, json=None):
        """
        @param path: API path, eg. domain
        @type path: string
        @param params: optional params for query
        @type params: dict
        """
        if params is None:
            params = {}

        ret = self.wrap(self._session.put, self._abs_url(path), data=params, json=json)
        return self._process(ret)

    def delete(self, path, params=None):
        """
        @param path: API path, eg. domain
        @type path: string
        @param params: optional params for query
        @type params: dict
        """
        if params is None:
            params = {}

        ret = self.wrap(self._session.delete, self._abs_url(path), params=params)
        return self._process(ret)

    def get_pretty(self, path, params=None):
        print(dumps(self.get(path, params=params), indent=4))

    def post_pretty(self, path, params=None, json=None):
        print(dumps(self.post(path, params=params, json=json), indent=4))

    def put_pretty(self, path, params=None, json=None):
        print(dumps(self.put(path, params=params, json=json), indent=4))

    def delete_pretty(self, path, params=None):
        print(dumps(self.delete(path, params=params), indent=4))
