from .value_container import ValueContainer


class SchemaValidationError(Exception):
    def __init__(self, errors):
        self.errors = errors


class SchemaField:
    def __init__(self, name: str, **kwargs):
        """

        :param name: str Name of the field.
        :param flags=None: list[str] Argparse flags.
        :param type=any: type Type of the field value.
        :param required=False: bool Mark the field as required.
        :param default=None: any Default value for the field.
        :param from_env=True: bool Parse the argument from system environment values.
        :param help=None: str Help text for the field.
        """
        self.name = name.lower().replace('-', '_')
        self.flags = kwargs.get('flags', None)
        self.type = kwargs.get('type', None)
        self.required = kwargs.get('required', False)
        self.default = kwargs.get('default', None)
        self.from_env = kwargs.get('from_env', True)
        self.action = kwargs.get('action', None)
        self.help = kwargs.get('help', None)

    def env_key(self, prefix: str = None):
        key = self.name.upper()
        if prefix is not None:
            key = prefix.upper() + '_' + key
        return key

    def build_arg(self):
        if self.flags is None:
            return None, None

        kwargs = {
            'default': self.default,
            'action': self.action,
            'help': self.help,
        }

        if self.type is not None:
            kwargs['type'] = self.type

        return self.flags, kwargs

    def validate_value(self, value):
        if value is None and self.required:
            return self.name + ' is a required value'
        if self.type is not None and value is not None and not isinstance(value, self.type):
            return self.name + ' has an invalid type of %s. Expected a type of %s' % (type(value), self.type)
        return None

    def __repr__(self):
        return '%s (flags=%s, type=%s, required=%s, default=%s, from_env=%s, help=%s)' % (
            self.name, self.flags, self.type, self.required, self.default, self.from_env, self.help)


class Schema:
    def __init__(self, *fields: SchemaField):
        self.fields = {}
        for field in fields:
            if not hasattr(field, 'name'):
                raise KeyError('Invalid field. Name is a required attribute')
            self.fields[field.name] = field

    def validate_values(self, values: ValueContainer):
        errors = []
        for key, field in self.fields.items():
            value = values.get(key, None)
            error = field.validate_value(value)
            if error is not None:
                errors.append(error)
        if len(errors) > 0:
            raise SchemaValidationError(errors)

    def build_args(self):
        args = []
        for key, field in self.fields.items():
            flags, props = field.build_arg()
            if flags is not None:
                args.append([flags, props])
        return args

    def __repr__(self):
        return 'Schema[%s]' % ', '.join([str(f) for k, f in self.fields.items()])
