import os
import sys
import argparse
import json
import logging
import logging.config
from ast import literal_eval
from importlib.util import spec_from_file_location, module_from_spec

from renki.common.utilities.module_loader import import_authentication_modules, import_modules, RenkiModuleError

from renki.common.conf import SchemaField, Configuration, SchemaValidationError

default_logging_config = {
    'version': 1,
    'disable_existing_loggers': True,
    'formatters': {
        'verbose': {
            'format': '%(levelname)s %(asctime)s %(module)s %(process)d %(thread)d %(message)s'
        },
        'simple': {
            '()': 'logging.Formatter',
            'format': '%(asctime)-20s %(levelname)s %(module)s %(message)s'
        }
    },
    'filters': {
    },
    'handlers': {
        'null': {
            'level': 'DEBUG',
            'class': 'logging.NullHandler',
        },
        'console': {
            'class': 'logging.StreamHandler',
            'formatter': 'verbose'
        }
    },
    'loggers': {
        'server': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'admin': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'INFO',
        },
        'migrations': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG'
        },
        'utils': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'ticket': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'INFO',
        },
        'module_database': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'module_dns_zone': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'module_domain': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'module_port': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'module_repository': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'module_dummy': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'tickets_done': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'RenkiSocket': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
    }
}


class Settings(Configuration):
    # TODO: Handle unknown arguments
    def __init__(self, *schema_fields):
        fields = []
        for field in schema_fields:
            if isinstance(field, tuple):
                fields.append(SchemaField(field[0], **field[1]))
            else:
                fields.append(field)

        super(Settings, self).__init__(*fields, env_var_prefix='RENKI')
        self.add_field('logging_config', fields=['--logging-config'], type=str, default=None)

    def _initialize_modules(self, context):
        if context is 'cli':
            return

        # Import authentication module
        if context is 'core':
            self.AUTH_MODULES = import_authentication_modules(self.AUTH_MODULES)

        # Import modules
        self.MODULES = import_modules(self.MODULES, context)

    def _configure_logging(self):
        if 'logging_config' in self and self.logging_config is not None:
            with open(self.logging_config) as json_data:
                config = json.load(json_data)
                logging.config.dictConfig(config)
        else:
            logging.config.dictConfig(default_logging_config)

    def parse(self, context: str, *value_dicts: dict, args=True, env_vars=True):
        try:
            super(Settings, self).parse(*value_dicts, args=args, env_vars=env_vars)
            self._initialize_modules(context)
            self._configure_logging()
        except SchemaValidationError as e:
            print('Failed to initialize due to following configuration errors:')
            for error in e.errors:
                print('* ' + error)
            sys.exit(1)
        except RenkiModuleError as e:
            print('Failed to initialize due to missing module package:', e)
            sys.exit(1)
        except Exception as e:
            print('Failed to initialize due to unexcepted error', e)
            sys.exit(1)
