import os
import unittest

from renki.common.conf import Configuration, SchemaField, SchemaValidationError


class TestConfParser(unittest.TestCase):
    def setUp(self):
        self.confParser = Configuration(
            SchemaField('debug',
                        flags=['-d', '--debug'],
                        default=False,
                        action='store_true',
                        help='Switch to enable debugging mode',),
            SchemaField('name',
                        flags=['-n', '--name'],
                        type=str,
                        required=True,
                        help='Name of the user',),
            SchemaField('doggo',
                        flags=['-y', '--doggo'],
                        from_env=False,
                        type=str,
                        help='Are you a doggo?'),
            SchemaField('count',
                        flags=['-i', '--count'],
                        type=int,
                        help='Name of the user',),
            SchemaField('my_list',
                        flags=['-l', '--my-list'],
                        type=list,
                        help='My list of stuff'),
            env_var_prefix='MY_PREFIX'
        )

    def test_should_parse_from_dict(self):
        self.confParser.parse({
            'DEBUG': True,
            'NAME': 'Test User'
        }, args=False, env_vars=False)
        self.assertTrue(self.confParser.DEBUG)
        self.assertEqual('Test User', self.confParser.NAME)

    def test_should_parse_only_once(self):
        self.test_should_parse_from_dict()
        with self.assertRaises(RuntimeError):
            self.confParser.parse(args=False)

    def test_should_raise_when_required_missing(self):
        with self.assertRaises(SchemaValidationError):
            self.confParser.parse({
                'DEBUG': True
            }, args=False, env_vars=False)

    def test_should_parse_from_env_vars(self):
        os.environ['MY_PREFIX_DEBUG'] = 'True'
        os.environ['MY_PREFIX_NAME'] = 'Test User'
        os.environ['MY_PREFIX_DOGGO'] = 'much doggo'
        os.environ['MY_PREFIX_COUNT'] = '10'
        os.environ['MY_PREFIX_MY_LIST'] = 'one,two,tree'
        self.confParser.parse(args=False)
        self.assertTrue('DEBUG' in self.confParser, 'Expected \'DEBUG\' to be defined on parsed config')
        self.assertTrue(self.confParser.DEBUG, 'Expected \'DEBUG\' to be true')
        self.assertFalse('DOGGO' in self.confParser, 'Expected \'DOGGO\' not to be parsed from env')
        self.assertTrue('COUNT' in self.confParser, 'Expected \'COUNT\' to be defined on parsed config')
        self.assertEqual(10, self.confParser.COUNT, 'Expected \'COUNT\' to equal to target value')
        self.assertTrue('MY_LIST' in self.confParser, 'Expected \'MY_LIST\' to be defined on parsed config')
        self.assertEqual(3, len(self.confParser.MY_LIST), 'Expected \'MY_LIST\' to contain three elements')

    def test_should_parse_from_args(self):
        args = ['-d', '-n', 'Test User', '--count', '10']
        self.confParser.parse(args=args)
        self.assertTrue('DEBUG' in self.confParser, 'Expected \'DEBUG\' to be defined on parsed config')
        self.assertTrue('NAME' in self.confParser, 'Expected \'NAME\' to be defined on parsed config')
        self.assertTrue('COUNT' in self.confParser, 'Expected \'COUNT\' to be defined on parsed config')

    def test_should_parse_from_json_file_from_args(self):
        args = ['-n', 'confTest', '--config', os.path.join(os.path.dirname(os.path.abspath(__file__)), 'testConfig.json')]
        self.confParser.parse(args=args)
        self.assertIn('CONFIG', self.confParser, 'Expected \'CONFIG\' to be defined on parsed config')
        self.assertEqual('config test', self.confParser.NAME, 'Expected \'NAME\' to equal to target value')

    def test_should_parse_from_json_file_from_env(self):
        os.environ['MY_PREFIX_CONFIG'] = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'testConfig.json')
        self.confParser.parse(args=False)
        self.assertIn('CONFIG', self.confParser, 'Expected \'CONFIG\' to be defined on parsed config')
        self.assertEqual('config test', self.confParser.NAME, 'Expected \'NAME\' to equal to target value')
