from abc import ABCMeta, abstractproperty


class AbstractPermission(object):
    __metaclass__ = ABCMeta

    _all_permissions = {}

    def __init__(self, name, description):
        if name in self._all_permissions.keys():
            raise Exception('Permission %s already defined' % name)

        self._name = name
        self._description = description
        self._all_permissions[self._name] = self

    @property
    def name(self):
        return self._name

    @property
    def description(self):
        return self._description

    @abstractproperty
    def is_global(self):
        pass

    @classmethod
    def get_permission_by_name(cls, name):
        if name in cls._all_permissions.keys():
            return cls._all_permissions[name]

    @classmethod
    def get_permissions(cls):
        return cls._all_permissions

    def __str__(self):
        return self._name


class GlobalPermission(AbstractPermission):
    """
    Permission that isn't bound to specific service but is more generic.
    For example permission to view or modify information of member.
    """
    def is_global(self):
        return True


class ServicePermission(AbstractPermission):
    """
    Permission that is bound to specific service.
    For example modifying members port on server1 or modifying members port on server2
    """
    def is_global(self):
        return False


def get_permission(name):
    """
    Get permission by name
    :param name: permission name
    :return: Permission if found, None otherwise
    """
    return AbstractPermission.get_permission_by_name(name)


def get_permissions():
    """
    Get dictionary containing all permissions
    :return: dictionary containing permissions
    """
    return AbstractPermission.get_permissions()
