from renki.core.lib.renkiapi import renkiapi
from renki.core.lib.exceptions import DoesNotExist
from renki.core.lib.auth.db import User
from renki.core.lib.utils import ok as ret_ok, conflict as ret_conflict, forbidden as ret_denied,\
    not_found as ret_not_found
from renki.core.lib.auth.basic_permissions import ViewUsers, EditUsers
from renki.core.lib.database.table import db
from flask import Blueprint

blueprint = Blueprint('users', __name__)


@renkiapi('GET', '/users',
          response={
              'type': 'array'
          }, blueprint=blueprint)
def users_list(identity):
    if not identity.has_permission(ViewUsers):
        return ret_denied('Not allowed')

    users = []
    for user in User.query.all():
        users.append({
            'id': user.id,
            'name': user.name,
            'auth_module': user.auth_module,
            'added': str(user.added)
        })

    return ret_ok(users)


@renkiapi('POST', '/users',
          json={
              'type': 'object',
              'properties': {
                  'name': {'type': 'string'},
                  'authModule': {'type': 'string'}
              },
              'required': ['name', 'authModule'],
              'additionalProperties': False
          },
          response={
              'type': 'object',
              'properties': {
                  'user_id': {'type': 'integer'}
              },
              'required': ['user_id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def users_add(identity, data):
    if not identity.has_permission(EditUsers):
        return ret_denied('Not allowed')

    name = data['name']
    auth_module = data['authModule']

    if User.query.filter(User.name == name, User.auth_module == auth_module).count():
        return ret_conflict('User %s with auth_module %s already exists' % (name, auth_module))

    user = User()
    user.name = name
    user.auth_module = auth_module
    user.save()

    db.session.commit()

    return ret_ok({'user_id': user.id})


@renkiapi('DELETE', '/users/<int:id>',
          url_params={
              'type': 'object',
              'properties': {
                  'id': {'type': 'integer'}
              },
              'required': ['id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def users_delete(identity, data):
    if not identity.has_permission(EditUsers):
        return ret_denied('Not allowed')

    user_id = int(data['id'])

    try:
        user = User.get(user_id)
    except DoesNotExist:
        return ret_not_found('User with id %i not found' % user_id)

    db.session.delete(user)
    db.session.commit()

    return ret_ok()
