import getpass

from flask.ext.script import Manager, Command, Option
from flask.ext.script.commands import InvalidCommand

from renki.core.lib.auth.db import BasicAuthUser
from renki.core.lib.commands.command_utils import *
from renki.core.lib.database.table import db
from renki.core.lib.exceptions import AlreadyExist

user_manager = Manager(usage="Manage users")


class ChangePassword(Command):
    """
    Change basic auth user password
    """
    option_list = (
        Option("username"),
    )

    def run(self, username):
        try:
            user = BasicAuthUser.query.filter(BasicAuthUser.name == username).one()
        except NoResultFound as e:
            raise InvalidCommand("User %s not found" % username)

        new_pass = getpass.getpass("New password: ")
        new_pass_again = getpass.getpass("New password again: ")

        if new_pass != new_pass_again:
            raise InvalidCommand("Passwords don't match!")

        user.set_password(new_pass)
        user.save()
        db.session.commit()

        print("Password set")


class AddUserCommand(Command):
    """
    Add new user
    """

    @property
    def option_list(self):
        return (
            Option('-m', '--module', dest="module",
                   default=list(get_auth_modules().keys())[0],
                   required=len(get_auth_modules().keys()) > 1,
                   help="Authentication module",
                   choices=get_auth_modules().keys()
                   ),
            Option('username')
        )

    def run(self, username, module):

        auth_module = get_auth_modules()[module]

        newpass = None
        if auth_module.require_password():
            newpass = getpass.getpass("New password: ")
            newpass_again = getpass.getpass("New password again: ")
            if newpass != newpass_again:
                raise InvalidCommand("Passwords don't match")

        try:
            user = auth_module.register_user(username, newpass)
        except AlreadyExist:
            raise InvalidCommand("User %s already created." % username)

        user.save()

        db.session.commit()


class ListUsers(Command):
    """
    List users
    """

    @property
    def option_list(self):
        return (
            Option('-m', '--module', dest="module", required=False,
                   help="Authentication module",
                   choices=get_auth_modules().keys()),
        )

    def run(self, module):

        print("{:<20} {:<20}".format("username", "module"))
        print("-"*80)
        for user in get_users():
            if module:
                if user.auth_module != module:
                    continue
            print("{:<20} {:<20}".format(user.name, user.auth_module))


def create_commands():
    user_manager.add_command("add", AddUserCommand)
    user_manager.add_command("list", ListUsers)
    user_manager.add_command("change-password", ChangePassword)
