from alembic.runtime.migration import MigrationContext
from alembic.autogenerate.api import produce_migrations, AutogenContext
from alembic.autogenerate.render import render_op
from alembic.operations import Operations
from alembic.operations.ops import UpgradeOps, DowngradeOps
from renki.core.lib import renki_settings as settings
from renki.core.lib.history_meta import Versioned
from renki.core.lib.exceptions import RenkiException
from renki.core.version import __version__
from renki.core.lib.database.basic_tables import Meta
from renki import core as renki_core
from mako.template import Template
from autopep8 import fix_code
from datetime import datetime
from sqlalchemy.exc import ProgrammingError, OperationalError
from sqlalchemy.orm.exc import NoResultFound
import os.path
import pkgutil
import importlib
import logging

logger = logging.getLogger('migrations')


class AutoPEP8Options:
    aggressive = True
    experimental = False
    ignore = False
    select = ''
    max_line_length = 120
    verbose = False
    line_range = False
    indent_size = 4
    pep8_passes = -1


AUTOGEN_OPTIONS = {
    'sqlalchemy_module_prefix': 'sa.',
    'alembic_module_prefix': 'op.',
    'render_item': None,
    'render_as_batch': False,
    'user_module_prefix': 'ext.'
}


class MigrationError(RenkiException):
    pass


MIGRATION_TEMPLATE = Template("""# Renki migrations - generated by Renki ${renki_version} on ${date}
# ${name} : ${comment}
import sqlalchemy as sa
import sqlalchemy_utils.types as ext

version = ${version}

def upgrade(op):
${upgrade_operations}

def downgrade(op):
${downgrade_operations}
""")


def make_migrations(name: str, comment: str, engine, metadata):
    """
    Create migrations for core and each module
    :param name: Name of the migration
    :param comment: Comment for the migration
    :param engine: SQLAlchemy engine that is compared against metadata
    :param metadata: Metadata of the target schema
    """
    context = MigrationContext.configure(engine)
    migrations = produce_migrations(context, metadata)

    # Determine which tables belong to modules
    tables_to_modules = {}
    for module in settings.MODULES:
        if not hasattr(module, 'tables'):
            continue

        for table in module.tables:
            tables_to_modules[table.__tablename__] = module

            # If table is versioned, take it into account as well
            if issubclass(table, Versioned):
                tables_to_modules[table.__tablename__ + '_history'] = module

    core_ops = ([], [])
    module_ops = {}

    # Group upgrade_ops to their corresponding modules
    for ops in migrations.upgrade_ops_list:
        for op in ops.ops:
            table_name = op.table_name

            if table_name not in tables_to_modules.keys():
                core_ops[0].append(op)
            else:
                module = tables_to_modules[table_name]

                if module not in module_ops:
                    module_ops[module] = ([], [])

                module_ops[module][0].append(op)

    # Group downgrade ops to their corresponding modules
    for ops in migrations.downgrade_ops_list:
        for op in ops.ops:
            table_name = op.table_name

            if table_name not in tables_to_modules.keys():
                core_ops[1].append(op)
            else:
                module = tables_to_modules[table_name]

                if module not in module_ops:
                    module_ops[module] = ([], [])

                module_ops[module][1].append(op)

    if len(core_ops[0]) > 0 or len(core_ops[1]) > 0:
        # Create migrations for core if needed
        create_migration(name, comment, renki_core, UpgradeOps(ops=core_ops[0]), DowngradeOps(ops=core_ops[1]))

    # Create migrations for each module
    for module, ops in module_ops.items():
        create_migration(name, comment, module, UpgradeOps(ops=ops[0]), DowngradeOps(ops=ops[1]))


def create_migration(name, comment, module, upgrade_ops, downgrade_ops):
    """

    :param name: Name of the version
    :param comment: Comment about what was changed
    :param module: module to use as root for searching for existing migrations
    :param upgrade_ops: Alembic UpgradeOps containing a list of operations to apply to upgrade
    :param downgrade_ops: Alembic DowngradeOps containing a list of operations to apply to downgrade
    """
    # Try to find package migrations under module or create it if it doesn't exist and determine version of
    # migration to be created
    try:
        module.migrations = importlib.import_module(module.__name__ + '.migrations')

        migrations = module.migrations
        migration_scripts = get_migration_scripts(os.path.dirname(migrations.__file__), migrations.__name__)
        migration_directory = os.path.dirname(migrations.__file__)
        if len(migration_scripts) > 0:
            next_version = migration_scripts[-1].version + 1
        else:
            next_version = 1
    except ImportError:
        migration_directory = os.path.dirname(module.__file__) + '/migrations'

        if not os.path.exists(migration_directory):
            os.makedirs(migration_directory)

        open(migration_directory + '/__init__.py', 'a').close()
        next_version = 1

    write_migrations(name,
                     comment,
                     next_version,
                     migration_directory + '/%04d_%s.py' % (next_version, name),
                     upgrade_ops,
                     downgrade_ops)


def get_migration_scripts(base_dir, base_package):
    migrations = []
    for importer, name, _ in pkgutil.iter_modules([base_dir], prefix=base_package + '.'):
        migration = importlib.import_module(name)
        if not hasattr(migration, 'version'):
            raise MigrationError('Migration %s is missing version' % migration.__name__)
        elif not hasattr(migration, 'upgrade'):
            raise MigrationError('Migration %s is missing upgrade method' % migration.__name__)
        elif not hasattr(migration, 'downgrade'):
            raise MigrationError('Migration %s is missing downgrade method')

        migrations.append(migration)

    migrations = sorted(migrations, key=lambda x: x.version, reverse=False)
    return migrations


def write_migrations(name, comment, version, file, upgrade_ops=None, downgrade_ops=None):
    """
    Write migrations to file
    :param name: Name of the migration
    :param comment: Comment of the migration
    :param version:
    :param file: File to write migrations to
    :param upgrade_ops: UpgradeOps containing operations to apply to upgrade from current version to version
    :param downgrade_ops: DowngradeOps containing operations to apply to downgrade from version to current version
    """
    ctx = AutogenContext(None, opts=AUTOGEN_OPTIONS)

    upgrade_contents = ''
    if upgrade_ops is None or len(upgrade_ops.ops) == 0:
        upgrade_contents = '    pass'
    else:
        for op in upgrade_ops.ops:
            lines = render_op(ctx, op)
            for line in lines:
                for row in str.split(line.strip(), '\n'):
                    upgrade_contents += '    %s\n' % row

    downgrade_contents = ''
    if downgrade_ops is None or len(downgrade_ops.ops) == 0:
        downgrade_contents = '    pass'
    else:
        for op in downgrade_ops.ops:
            lines = render_op(ctx, op)
            for line in lines:
                for row in str.split(line.strip(), '\n'):
                    downgrade_contents += '    %s\n' % row

    code = MIGRATION_TEMPLATE.render(renki_version=__version__,
                                     date=datetime.now(),
                                     name=name,
                                     comment=comment,
                                     version=version,
                                     upgrade_operations=upgrade_contents,
                                     downgrade_operations=downgrade_contents)

    formatted = fix_code(code, options=AutoPEP8Options)
    with open(file, 'w') as f:
        f.write(formatted)


def run_migrations(db, module_name=None, target_version=None):
    conn = db.engine.connect()
    context = MigrationContext.configure(conn)
    op = Operations(context)

    if target_version is not None:
        target_version = int(target_version)

    logger.info('Determining which migrations need to be run')
    if module_name is None or module_name == 'core':
        try:
            meta_version = Meta.query.filter(Meta.variable == 'core_schema_version').one()
            current_version = meta_version.int_value
        except (ProgrammingError, NoResultFound, OperationalError):
            current_version = 0

        updated_version = apply_migrations(renki_core, op, current_version, target_version)
        db.session.commit()

        try:
            meta_version = Meta.query.filter(Meta.variable == 'core_schema_version').one()
        except NoResultFound:
            meta_version = Meta()
            meta_version.variable = 'core_schema_version'

        meta_version.int_value = updated_version
        meta_version.save()
        db.session.commit()

    for module in settings.MODULES:
        if module_name is None or module.name == module_name:
            try:
                meta_version = Meta.query.filter(Meta.variable == module.name + '_schema_version').one()
                current_version = meta_version.int_value
            except (ProgrammingError, NoResultFound, OperationalError):
                current_version = 0

            updated_version = apply_migrations(module, op, current_version, target_version)
            db.session.commit()

            try:
                meta_version = Meta.query.filter(Meta.variable == module.name + '_schema_version').one()
            except NoResultFound:
                meta_version = Meta()
                meta_version.variable = module.name + '_schema_version'

            meta_version.int_value = updated_version
            meta_version.save()
            db.session.commit()


def apply_migrations(module, op, current_version, target_version=None):
    """
    Apply migrations for module from current version to target version

    :param module:
    :param op: Alembic operations bound to context
    :param current_version: Current version of the module's migrations
    :param target_version: Target version of the module's migrations
    :return: Last successful version
    """
    migration_module = module.__name__
    if 'module' in migration_module:
        migration_module = '.'.join(migration_module.split('.')[:-1])

    try:
        migrations = importlib.import_module(migration_module + '.migrations')
    except ImportError:
        logger.warn('No migrations found in ' + migration_module)
        return

    migration_scripts = get_migration_scripts(os.path.dirname(migrations.__file__), migrations.__name__)

    # If target version isn't defined, use the latest version instead
    if target_version is None:
        target_version = migration_scripts[-1].version

    # TODO : add proper transactions if needed
    if target_version > current_version:
        for i in range(0, len(migration_scripts)):
            if current_version < migration_scripts[i].version <= target_version:
                logger.info('Updating %s to %s' % (migration_module, migration_scripts[i].version))
                migration_scripts[i].upgrade(op)
    elif target_version < current_version:
        for i in range(len(migration_scripts) - 1, -1, -1):
            if target_version < migration_scripts[i].version <= current_version:
                logger.info('Downgrading %s to %s' % (migration_module, migration_scripts[i].version - 1))
                migration_scripts[i].downgrade(op)
    else:
        logger.info('%s is at correct version' % migration_module)

    return target_version
