from flask.ext.script import Manager, Option, Command
from flask.ext.script.commands import InvalidCommand

from renki.core.lib.commands.command_utils import *
from renki.core.lib.database.table import db

import logging

logger = logging.getLogger("server_command")

server_manager = Manager(usage="Manage servers")


class ListServers(Command):
    """
    List servers in database
    """

    def run(self):
        print("{:<40} {:<50}".format("name", "services"))
        print("-"*80)
        for server in sorted(get_servers(), key=lambda x: x.name):
            print("{:<40} {:<50}".format(server.name, ', '.join([x.name for x in server.services])))


class AddServer(Command):
    """
    Add server
    """

    option_list = (
        Option('name', help="Server name"),
    )

    def run(self, name):
        server_obj = get_server(name)

        if server_obj:
            raise InvalidCommand("Server already created")

        server_obj = get_or_create_server(name)

        server_obj.save()

        db.session.commit()

        logger.info("Server %s added using commandline" % name)


class RemoveServer(Command):
    """
    Remove server
    """

    option_list = (
        Option('server', help="Server name"),
    )

    def run(self, server):
        server_obj = get_server(server)
        if not server_obj:
            raise InvalidCommand("Server %s not found" % server)

        db.session.delete(server_obj)
        db.session.commit()

        logger.info("Server %s removed using commandline" % server)


server_manager.add_command("list", ListServers)
server_manager.add_command("add", AddServer)
server_manager.add_command("remove", RemoveServer)
