from renki.core.lib.renkiapi import renkiapi
from renki.core.lib.utils import ok as ret_ok, forbidden as ret_denied, not_found as ret_not_found, error as ret_error
from renki.core.lib.database.basic_tables import Server
from renki.core.lib.communication.ticket_tables import Ticket
from renki.core.lib.auth.basic_permissions import SrvFetchTickets
from renki.core.lib.database.table import db
from sqlalchemy.orm.exc import NoResultFound
from sqlalchemy.orm import with_polymorphic
from datetime import datetime
from flask import Blueprint

blueprint = Blueprint('tickets', __name__)


@renkiapi('GET', '/tickets/<server>',
          url_params={
              'type': 'object',
              'properties': {
                  'server': {'type': 'string'}
              },
              'required': ['server'],
              'additionalProperties': False
          },
          response={
              'type': 'object',
              'properties': {
                  'tickets': {'type': 'array'},
                  'next_poll': {'type': 'integer'}
              },
              'required': ['tickets', 'next_poll'],
              'additionalProperties': False

          }, blueprint=blueprint)
def get_tickets(identity, data):
    if not identity.has_permission(SrvFetchTickets):
        return ret_denied()

    server_name = data['server']

    try:
        server = Server.query.filter(Server.name == server_name).one()
    except NoResultFound:
        return ret_not_found('Server with name %s does not exist' % server_name)

    # Find tickets which haven been acquired yet and are going to the server
    # TODO : also give tickets that have been acquired long time ago but haven't been done
    tickets = db.session.query(with_polymorphic(Ticket, '*')).filter(Ticket.server == server,
                                                                     Ticket.acquired.is_(None)).all()

    ret_tickets = []

    # Mark tickets as acquired
    for ticket in tickets:
        print(ticket.ticket_module)
        ticket.acquired = datetime.now()
        ticket.save()

        ticket_dict = ticket.to_dict()
        if ticket.unix_user is not None:
            ticket_dict.update({'unix_user_name': ticket.unix_user.name})

        ret_tickets.append(ticket_dict)

    db.session.commit()

    return ret_ok({'tickets': ret_tickets, 'next_poll': 60})


@renkiapi('POST', '/tickets/<int:id>',
          url_params={
              'type': 'object',
              'properties': {
                  'id': {'type': 'integer'}
              },
              'required': ['id'],
              'additionalProperties': False
          },
          json={
              'type': 'object',
              'properties': {
                  'comment': {'type': 'string'},
                  'error': {'type': 'string'}
              },
              'additionalProperties': False
          }, blueprint=blueprint)
def finish_ticket(identity, data):
    if not identity.has_permission(SrvFetchTickets):
        return ret_denied()

    ticket_id = int(data['id'])
    comment = data['comment'] if 'comment' in data else None
    error = data['error'] if 'error' in data else None

    ticket = Ticket.get(ticket_id)

    if comment is not None and error is not None:
        return ret_error('Both comment and error should not be defined')

    if comment is None and error is None:
        return ret_error('Either comment or error should be defined')

    # TODO: make sure that state of ticket is acquired and that it's not finished

    if comment is not None:
        ticket.done = datetime.utcnow()
        ticket.comment = comment
    else:
        ticket.failed = datetime.utcnow()
        ticket.error = error

    ticket.save()

    db.session.commit()

    return ret_ok()
