"""
This file is part of Renki project

Base utils for testing
"""
import unittest
import json
import logging
import logging.config
from webtest import TestApp
from renki.core.lib.renki_app import create_app
from renki.core.lib.database.basic_tables import Server, Service
from renki.core.lib.auth.authentication_modules.basic import BasicAuthenticationModule
from renki.core.lib.database.table import db

from renki.core.context import settings

logger = logging.getLogger()
logger.setLevel(logging.ERROR)

logging.config.dictConfig({'version': 1, 'disable_existing_loggers': True})

with open('test-config.json') as json_data:
    test_settings = json.load(json_data)
    settings.load('core', test_settings)


class TestUser(object):
    def __init__(self, name, password, session_key, user):
        self.name = name
        self.session_key = session_key
        self.password = password
        self.user = user


class BasicTest(unittest.TestCase):
    def setUp(self):
        config = {
            'SQLALCHEMY_DATABASE_URI': settings.CONNECTION_STRING
        }
        self.app = create_app(config)
        self.app_context = self.app.app_context()
        self.app_context.push()
        db.create_all()

        self.test_app = TestApp(self.app)
        self._users = {}
        self._auth_token = None

    def tearDown(self):
        db.session.rollback()
        db.drop_all()
        db.get_engine(self.app).dispose()
        self.app_context.pop()

    def _get_error(self, item):
        error = None
        try:
            if 'error' in item.json:
                error = item.json['error']
        except (AttributeError, KeyError):
            pass
        return error

    def auth(self, user, password):
        """
        Authenticate user and return authentication key
        """
        args = {'username': user, 'password': password}
        s = self.test_app.post('/login', params=args, status="*")
        self._auth_token = s.json['authToken']

        return s.json['authToken']

    def user(self, name):
        """
        Create user with name `name` and permissions `permissions`
        return authenticated session
        """
        if name not in self._users:
            pw = 'password'

            user = BasicAuthenticationModule.register_user(name, pw)
            db.session.commit()

            session_key = self.auth(user=name, password=pw)
            u = TestUser(name=name, password=pw, session_key=session_key, user=user)
            self._users[name] = u
        return self._users[name]

    def get_or_create_server(self, server_name):
        """
        Find server with matching name or create new one if server isn't found
        :param server_name: name of the server (f
        :return: server identified by server_name
        """
        if not Server.query.filter(Server.name == server_name).all():
            server = Server()
            server.name = server_name
            db.session.add(server)
            return server

        return Server.query.filter(Server.name == server_name).one()

    def get_or_create_service(self, service_name, service_type, service_servers=None):
        """
        Find service with matching name or create new one if service isn't found
        :param service_name: name of the service (for example. Lakka_port)
        :param service_type: type of the service (for example. port)
        :param service_servers: servers that handle tasks related to the service (for example. [hilla, lakka])
        :return: service identified by service_name
        """
        if not Service.query.filter(Service.name == service_name).all():
            service = Service()
            service.name = service_name
            service.service_type = service_type
            service.servers = service_servers
            db.session.add(service)
            return service

        return Service.query.filter(Service.name == service_name).one()

    def get(self, url, params=None, headers=None, status=None, extra_environ=None, expect_errors=False, xhr=False):
        if self._auth_token is not None:
            if headers is None:
                headers = {}
            headers['Authorization'] = self._auth_token

        return self.test_app.get(url, params, headers=headers, extra_environ=extra_environ, status=status,
                                 expect_errors=expect_errors, xhr=xhr)

    def post(self, url, params=None, headers=None, status=None, extra_environ=None, expect_errors=False):
        if self._auth_token is not None:
            if headers is None:
                headers = {}
            headers['Authorization'] = self._auth_token

        return self.test_app.post_json(url, params, headers=headers, extra_environ=extra_environ, status=status,
                                       expect_errors=expect_errors)

    def delete(self, url, params=None, headers=None, status=None, extra_environ=None, expect_errors=False,
               content_type=None):
        if self._auth_token is not None:
            if headers is None:
                headers = {}
            headers['Authorization'] = self._auth_token

        return self.test_app.delete_json(url, params, headers=headers, extra_environ=extra_environ, status=status,
                                         expect_errors=expect_errors, content_type=content_type)

    def put(self, url, params=None, headers=None, status=None, extra_environ=None, expect_errors=False):
        if self._auth_token is not None:
            if headers is None:
                headers = {}
            headers['Authorization'] = self._auth_token

        return self.test_app.put_json(url, params, headers=headers, extra_environ=extra_environ, status=status,
                                      expect_errors=expect_errors)
