from flask.ext.script import Command, Manager, Option
from flask.ext.script.commands import InvalidCommand

from renki.core.lib.auth.db import Permission

from renki.core.lib.commands.command_utils import *
from renki.core.lib.database.table import db

import logging

logger = logging.getLogger("permission_command")

permission_manager = Manager(usage="Manage permissions")
global_permission_manager = Manager(usage="Global permissions")
service_permission_manager = Manager(usage="Service permissions")


class SyncGlobalPermissions(Command):
    """
    Sync global permissions to database
    """
    option_list = (
        Option("-n", '--name', help="Sync only permission matching name", dest="name", required=False),
    )

    def run(self, name):
        if name:
            permission = get_permission(name)
            if not permission:
                raise InvalidCommand("Cannot find global permission %s" % permission)
            perm = get_or_create_global_permission(permission)
            perm.save()
        else:
            sync_global_permissions()

        db.session.commit()

        logger.warning("Global permissions synchronized using commandline")


class ListGlobalPermissions(Command):
    """
    List global permissions from database
    """

    def run(self):
        print("{:<40} {:<50}".format("permission", "description"))
        print("-"*80)
        for permission in Permission().query.all():
            if not permission.is_global:
                continue
            print("{:<40} {:<50}".format(permission.name, permission.description))


class AddServicePermission(Command):
    """
    Add service permission to database
    """
    option_list = (
        Option("-d", '--description', help="Description", dest="description", required=False, default=""),
        Option("service"),
        Option("permission"),
    )

    def run(self, service, permission, description):
        permission_obj = get_permission(permission)
        if not permission_obj or permission_obj.is_global():
            raise InvalidCommand("Cannot find service permission %s" % permission)

        service_obj = get_service(service)
        if not permission_obj:
            raise InvalidCommand("Cannot find service %s" % service)

        if not description:
            description = "%s in %s" % (permission_obj.description, service_obj.name)

        perm = get_or_create_service_permission(permission_obj, description, service_obj)
        perm.save()

        db.session.commit()

        logger.warning("Service permission %s for service %s added using commandline" % (permission, service))


class ListServicePermissions(Command):
    """
    List service permissions from database
    """

    def run(self):
        print("{:<40} {:<30} {:<50}".format("permission", "service", "description"))
        print("-"*80)
        for permission in Permission().query.all():
            if permission.is_global:
                continue
            service = get_service_by_id(permission.service_id)
            print("{:<40} {:<30} {:<50}".format(permission.name, service.name, permission.description))


class ListAvailableServicePermissions(Command):
    """
    List available service permissions
    """

    def run(self):
        print("{:<40} {:<50}".format("permission", "description"))
        print("-"*80)
        for permission_name, permission in get_permissions().items():
            if permission.is_global():
                continue
            print("{:<40} {:<50}".format(permission.name, permission.description))


class RemoveServicePermission(Command):
    """
    Remove service permission from database
    """
    option_list = (
        Option("service"),
        Option("permission"),
    )

    def run(self, service, permission):
        permission_obj = get_permission(permission)
        if not permission_obj or permission_obj.is_global():
            raise InvalidCommand("Cannot find service permission %s" % permission)

        service_obj = get_service(service)
        if not permission_obj:
            raise InvalidCommand("Cannot find service %s" % service)

        perm = get_service_permission(permission_obj, service_obj)

        if not perm:
            raise InvalidCommand("Permission does not exist")

        db.session.delete(perm)

        db.session.commit()

        logger.warning("Service permission %s for service %s removed using commandline" % (permission, service))


global_permission_manager.add_command("list", ListGlobalPermissions)
global_permission_manager.add_command("sync", SyncGlobalPermissions)
service_permission_manager.add_command("list", ListServicePermissions)
service_permission_manager.add_command("keys", ListAvailableServicePermissions)
service_permission_manager.add_command("add", AddServicePermission)
service_permission_manager.add_command("remove", RemoveServicePermission)

permission_manager.add_command("global", global_permission_manager)
permission_manager.add_command("service", service_permission_manager)
