from renki.core.lib.renkiapi import renkiapi
from renki.core.lib.exceptions import DoesNotExist
from renki.core.lib.auth.db import AuthTokens, UserToMember
from renki.core.lib.utils import ok as ret_ok, unauthorized as ret_noauth, conflict as ret_conflict,\
    forbidden as ret_denied, not_found as ret_not_found
from renki.core.lib.auth.basic_permissions import ViewUserMembers, EditUserMembers
from renki.core.lib.database.table import db
from flask import Blueprint

blueprint = Blueprint('user_members', __name__)


@renkiapi('GET', '/user_members',
          response={
              'type': 'object',
              'properties': {
                  'user_members': {'type': 'array'}
              },
              'required': ['user_members'],
              'additionalProperties': False
          }, blueprint=blueprint)
def user_members_list(identity):
    if not identity.has_permission(ViewUserMembers):
        return ret_denied('Not allowed')

    user_members = {}
    for user_member in UserToMember.query().all():
        user_members[user_member.id] = {
            'user_id': user_member.user_id,
            'member_id': user_member.member_id
        }

    return ret_ok(user_members)


@renkiapi('POST', '/user_members',
          json={
              'type': 'object',
              'properties': {
                  'user_id': {'type': 'integer'},
                  'member_id': {'type': 'integer'}
              },
              'required': ['user_id', 'member_id'],
              'additionalProperties': False
          },
          response={
              'type': 'object',
              'properties': {
                  'user_id': {'type': 'integer'}
              },
              'required': ['user_id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def user_members_add(auth_token, data):
    try:
        identity = AuthTokens.get_token(auth_token)
    except DoesNotExist:
        return ret_noauth('Not authenticated')

    if not identity.has_permission(EditUserMembers):
        return ret_denied('Not allowed')

    user_id = int(data['user_id'])
    member_id = int(data['member_id'])

    if UserToMember.query().filter(UserToMember.user_id == user_id, UserToMember.member_id == member_id).count():
        return ret_conflict('User-Member connection already exists from user %i to member %i' % (user_id, member_id))

    user_member = UserToMember()
    user_member.user_id = user_id
    user_member.member_id = member_id
    user_member.save()

    db.session.commit()

    return ret_ok({'user_member_id': user_member.id})


@renkiapi('DELETE', '/user_members/<int:user_member_id>',
          url_params={
              'type': 'object',
              'properties': {
                  'user_member_id': {'type': 'integer'}
              },
              'required': ['user_member_id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def user_members_delete(auth_token, user_member_id):
    try:
        identity = AuthTokens.get_token(auth_token)
    except DoesNotExist:
        return ret_noauth('Not authenticated')

    if not identity.has_permission(EditUserMembers):
        return ret_denied('Not allowed')

    try:
        user_member = UserToMember.get(user_member_id)
    except DoesNotExist:
        return ret_not_found('User-Member with id %i not found' % user_member_id)

    db.session.delete(user_member)
    db.session.commit()

    return ret_ok()
