import logging
from abc import ABCMeta, abstractclassmethod
from sqlalchemy.orm.exc import NoResultFound
from sqlalchemy import and_
from renki.core.lib.auth.db import User, AuthTokens, UserToMember
from renki.core.lib.exceptions import AuthenticationFailed, DoesNotExist, AlreadyExist
from renki.core.lib.database.table import db


logger = logging.getLogger('Authentication')


class AuthenticationModule(object):
    __metaclass__ = ABCMeta
    NAME = "NotNamed"

    @classmethod
    def register_user(cls, username: str, password: str, params: dict=None):
        """
        Create user `username` with password `password`
        :param username: Username of the new user
        :param password: Password of the new user
        :param params: Parameters associated with user (eg. Full Name, Address)
        """
        # Verify that user doesn't exist yet in the local database
        if User.query.filter(and_(User.auth_module == cls.NAME, User.name == username)).count() != 0:
            raise AlreadyExist("User %s already exists in module %s" % (username, cls.NAME))

        cls._register_user(username, password, params)

        user = User()
        user.name = username
        user.auth_module = cls.NAME
        db.session.add(user)

        return user

    @classmethod
    def authenticate(cls, username: str, password: str):
        """
        Authenticate against backend
        :return: key associated with current session
        """
        logger.debug("Authenticating user %s against module %s" % (username, cls.NAME))
        try:
            user = User.query.filter(and_(User.auth_module == cls.NAME, User.name == username)).one()
        except NoResultFound:
            raise AuthenticationFailed("Invalid username or password")

        result = cls._authenticate(username, password)

        # TODO: Implement two factor authentication

        # Generate identity
        identity = AuthTokens.add_token(user=user, token=result['token'], expires=result['expires'])

        query = UserToMember.query.filter(UserToMember.user_id == user.id)
        if query.count() == 1:
            identity.member_id = query.one().member_id
            # identity.save()

        db.session.commit()

        # TODO: Return a serializable identity context
        return {'authToken': identity.token}

    @classmethod
    def validate_token(cls, token: str):
        """
        Check if key is valid authentication key
        """

        # If local database doesn't contain the key it's definitely invalid
        try:
            AuthTokens.query.filter(AuthTokens.token == token).one()
        except DoesNotExist:
            raise

        # If local database has the key try to validate against backend
        cls._validate_token(token)

    @classmethod
    def _register_user(cls, username: str, password: str, params: dict):
        raise NotImplemented

    @classmethod
    def _authenticate(cls, username: str, password: str):
        raise NotImplemented

    @classmethod
    def _validate_token(cls, token: str):
        raise NotImplemented

    @abstractclassmethod
    def require_password(cls):
        pass
