from flask.ext.script import Manager, Option, Command
from flask.ext.script.commands import InvalidCommand

from renki.core.lib.commands.command_utils import *
from renki.core.lib.database.table import db
from renki.core.lib.auth.permissions import get_permission

from abc import ABCMeta, abstractclassmethod

import logging

logger = logging.getLogger("permission_group_command")

permission_group_manager = Manager(usage="Manage Permission groups")


def get_permission_const(name):
    permission_const = get_permission(name)
    if not permission_const:
        raise InvalidCommand("No permission named %s" % name)
    return permission_const


def get_service_command(name):
    service = get_service(name)
    if not service:
        raise InvalidCommand("Service %s not found" % name)
    return service


class AddPermissionGroup(Command):
    """
    Add permission group
    """

    option_list = (
        Option("-d", "--description", help="Description", dest="description", required=False, default=""),
        Option("name", help="Permission group name"),
    )

    def run(self, name, description):
        try:
            self.get_permission_group(name)
        except InvalidCommand:
            pass
        else:
            raise InvalidCommand("%s permission group %s already created" % (self.TYPE, name))

        group = self.add_permission_group(name, description)

        group.save()

        db.session.commit()

        logger.warning("%s permission group %s added using commandline" % (self.TYPE, name))


class ListPermissionGroup(Command):
    """
    List user permission group
    """

    def run(self):
        print("{:<40} {:<50}".format("name", "description"))
        print("-"*80)
        for group in self.get_permission_groups():
            print("{:<40} {:<50}".format(group.name, group.description))


class RemovePermissionGroup(Command):
    """
    Remove permission group
    """

    option_list = (
        Option("name", help="Permission group name"),
    )

    def run(self, name):
        permission_group = self.get_permission_group(name)

        db.session.delete(permission_group)
        db.session.commit()

        logger.warning("%s permission group %s removed using commandline" % (self.TYPE, name))


class PermissionGroupListPermissions(Command):
    """
    List permission group permissions
    """

    option_list = (
        Option("name", help="Permission group name"),
    )

    def run(self, name):

        permission_group = self.get_permission_group(name)

        print("{:<40} {:<25} {:<50}".format("name", "service",  "description"))
        print("-"*80)
        for permission in sorted(permission_group.permissions, key=lambda x: x.name):
            service = get_service_by_id(permission.service_id)
            print("{:<40} {:<25} {:<50}".format(permission.name,
                                                service.name if service else "", permission.description))


class PermissionGroupAddPermission(Command):
    """
    Add permission to permission group
    """

    option_list = (
        Option('-s', '--service', help="Add permission to service", dest="service_name", required=False),
        Option("permission", help="Permission name"),
        Option("group", help="Group name"),
    )

    def run(self, group, permission, service_name):

        permission_group = self.get_permission_group(group)
        permission_const = get_permission_const(permission)

        if service_name:
            service = get_service_command(service_name)
            permission_item = get_service_permission(permission_const, service)

        else:
            permission_item = get_global_permission(permission_const)

        permission_group.add_permission(permission_item)
        permission_group.save()
        db.session.commit()

        logger.warning("Permission %s added to %s permission group %s using commandline" % (permission, self.TYPE,
                                                                                            group))


class PermissionGroupRemovePermission(Command):
    """
    Remove permission from permission group
    """

    option_list = (
        Option('-s', '--service', help="Service name", dest="service_name", required=False),
        Option("permission", help="Permission name"),
        Option("group", help="Group name"),
    )

    def run(self, group, permission, service_name):

        permission_group = self.get_permission_group(group)
        permission_const = get_permission_const(permission)

        if service_name:
            service = get_service_command(service_name)
            permission_item = get_service_permission(permission_const, service)

        else:
            permission_item = get_global_permission(permission_const)

        permission_group.remove_permission(permission_item)
        permission_group.save()
        db.session.commit()

        logger.warning("Permission %s removed from %s permission group %s using commandline" % (permission, self.TYPE,
                                                                                                group))


class PermissionGroupListMembers(Command):
    """
    List permission group members
    """

    option_list = (
        Option("name", help="Permission group name"),
    )

    def run(self, name):

        permission_group_members = self.get_members(name)

        print("{:<80}".format("name"))
        print("-"*80)
        for member in sorted(permission_group_members, key=lambda x: self.format_member(x)):
            print("{:<80}".format(self.format_member(member)))


class PermissionGroupAddMember(Command):
    """
    Add member to permission group
    """
    option_list = (
        Option("member", help="Member name"),
        Option("name", help="Permission group name"),
    )

    def run(self, member, name):

        group = self.get_permission_group(name)
        member_obj = self.find_member(member)
        group.add_member(member_obj)

        group.save()
        db.session.commit()

        logger.warning("Member %s added to %s permission group %s using commandline" % (member, self.TYPE, name))


class PermissionGroupRemoveMember(Command):
    """
    Add member to permission group
    """
    option_list = (
        Option("member", help="Member name"),
        Option("name", help="Permission group name"),
    )

    def run(self, member, name):

        group = self.get_permission_group(name)
        member_obj = self.find_member(member)
        self.check_is_member(group, member_obj)
        group.remove_member(member_obj)

        group.save()
        db.session.commit()

        logger.warning("Member %s removed from %s permission group %s using commandline" % (member, self.TYPE, name))


class PermissionGroupCommand(object):
    __classmeta__ = ABCMeta

    @abstractclassmethod
    def get_permission_group(cls, name):
        raise NotImplementedError("TODO")

    @abstractclassmethod
    def get_permission_groups(cls):
        raise NotImplementedError("TODO")

    @abstractclassmethod
    def add_permission_group(cls, name, description):
        raise NotImplementedError("TODO")

    @abstractclassmethod
    def get_members(cls, name):
        raise NotImplementedError("TODO")

    @classmethod
    def format_member(cls, member):
        return member.name

    @abstractclassmethod
    def check_is_member(cls, group, name):
        raise NotImplementedError("TODO")

    @abstractclassmethod
    def find_member(cls, name):
        raise NotImplementedError("TODO")

# UserPermissionGroup


class UserPermissionGroupCommand(PermissionGroupCommand):
    TYPE = "User"

    @classmethod
    def get_permission_group(cls, name):
        permission_group = get_user_permission_group(name)
        if not permission_group:
            raise InvalidCommand("User permission group %s does not exist" % name)
        return permission_group

    @classmethod
    def get_permission_groups(cls):
        return get_user_permission_groups()

    @classmethod
    def add_permission_group(cls, name, description):
        return add_user_permission_group(name, description)

    @classmethod
    def get_members(cls, name):
        return cls.get_permission_group(name).users

    @classmethod
    def check_is_member(cls, group, member):
        if member not in group.users:
            raise InvalidCommand("User %s not in group %s" % (member.name, group.name))

    @classmethod
    def find_member(cls, name):
        user = get_user(name)
        if not user:
            raise InvalidCommand("User %s not found" % name)
        return user


class UserPermissionGroupRemovePermission(PermissionGroupRemovePermission, UserPermissionGroupCommand):
    """
    Remove permission from user permission group
    """
    get_permission_group = UserPermissionGroupCommand.get_permission_group


class UserPermissionGroupAddPermission(PermissionGroupAddPermission, UserPermissionGroupCommand):
    """
    Add permission to user permission group
    """
    pass


class UserPermissionGroupListPermissions(PermissionGroupListPermissions, UserPermissionGroupCommand):
    """
    List user permission group permissions
    """
    pass


class ListUserPermissionGroup(ListPermissionGroup, UserPermissionGroupCommand):
    """
    List user permission group
    """


class RemoveUserPermissionGroup(RemovePermissionGroup, UserPermissionGroupCommand):
    """
    Remove user permission group
    """


class AddUserPermissionGroup(AddPermissionGroup, UserPermissionGroupCommand):
    """
    Add user permission group
    """


class UserPermissionGroupListMembers(PermissionGroupListMembers, UserPermissionGroupCommand):
    """
    List user permission group members
    """


class UserPermissionGroupAddMember(PermissionGroupAddMember, UserPermissionGroupCommand):
    """
    Add member to user permission group members
    """


class UserPermissionGroupRemoveMember(PermissionGroupRemoveMember, UserPermissionGroupCommand):
    """
    Remove member to user permission group members
    """


user_permission_group_manager = Manager(usage="Manage user permission groups")
user_permission_group_manager.add_command("add", AddUserPermissionGroup)
user_permission_group_manager.add_command("remove", RemoveUserPermissionGroup)
user_permission_group_manager.add_command("list", ListUserPermissionGroup)
user_permission_group_manager.add_command("list-permissions", UserPermissionGroupListPermissions)
user_permission_group_manager.add_command('add-permission', UserPermissionGroupAddPermission)
user_permission_group_manager.add_command('remove-permission', UserPermissionGroupRemovePermission)
user_permission_group_manager.add_command('list-members', UserPermissionGroupListMembers)
user_permission_group_manager.add_command('add-member', UserPermissionGroupAddMember)
user_permission_group_manager.add_command('remove-member', UserPermissionGroupRemoveMember)


permission_group_manager.add_command("user", user_permission_group_manager, help="User permission group")

# UserMemberPermissionGroup


class UserMemberPermissionGroupCommand(PermissionGroupCommand):
    TYPE = "User member"

    @classmethod
    def get_permission_group(cls, name):
        permission_group = get_user_member_group(name)
        if not permission_group:
            raise InvalidCommand("User member permission group %s does not exist" % name)
        return permission_group

    @classmethod
    def get_permission_groups(cls):
        return get_user_member_groups()

    @classmethod
    def add_permission_group(cls, name, description):
        return add_user_member_group(name, description)

    @classmethod
    def get_members(cls, name):
        return cls.get_permission_group(name).user_members

    @classmethod
    def format_member(cls, member):
        member_obj = Member().query.filter(Member.id == member.member_id).one()
        user_obj = User().query.filter(User.id == member.user_id).one()
        return "%d, Member #%d to %s" % (member.id, member_obj.id, user_obj.name)

    @classmethod
    def check_is_member(cls, group, member):
        if member not in group.user_members:
            raise InvalidCommand("User member id %s not in group %s" % (member.id, group.name))

    @classmethod
    def find_member(cls, _id):
        user_member = get_user_member(_id)
        if not user_member:
            raise InvalidCommand("User member id %s not found" % _id)
        return user_member


class UserMemberPermissionGroupRemovePermission(PermissionGroupRemovePermission, UserMemberPermissionGroupCommand):
    """
    Remove permission from user member permission group
    """


class UserMemberPermissionGroupAddPermission(PermissionGroupAddPermission, UserMemberPermissionGroupCommand):
    """
    Add permission to user member permission group
    """
    pass


class UserMemberPermissionGroupListPermissions(PermissionGroupListPermissions, UserMemberPermissionGroupCommand):
    """
    List user member permission group permissions
    """
    pass


class ListUserMemberPermissionGroup(ListPermissionGroup, UserMemberPermissionGroupCommand):
    """
    List user member permission group
    """


class RemoveUserMemberPermissionGroup(RemovePermissionGroup, UserMemberPermissionGroupCommand):
    """
    Remove user member permission group
    """


class AddUserMemberPermissionGroup(AddPermissionGroup, UserMemberPermissionGroupCommand):
    """
    Add user member permission group
    """


class UserMemberPermissionGroupListMembers(PermissionGroupListMembers, UserMemberPermissionGroupCommand):
    """
    List user member permission group members
    """


class UserMemberPermissionGroupAddMember(PermissionGroupAddMember, UserMemberPermissionGroupCommand):
    """
    Add member to user member permission group
    """


class UserMemberPermissionGroupRemoveMember(PermissionGroupRemoveMember, UserMemberPermissionGroupCommand):
    """
    Add member to user member permission group
    """


user_member_permission_group_manager = Manager(usage="Manage user to member permission groups")
user_member_permission_group_manager.add_command("add", AddUserMemberPermissionGroup)
user_member_permission_group_manager.add_command("remove", RemoveUserMemberPermissionGroup)
user_member_permission_group_manager.add_command("list", ListUserMemberPermissionGroup)
user_member_permission_group_manager.add_command("list-permissions", UserMemberPermissionGroupListPermissions)
user_member_permission_group_manager.add_command('add-permission', UserMemberPermissionGroupAddPermission)
user_member_permission_group_manager.add_command('remove-permission', UserMemberPermissionGroupRemovePermission)
user_member_permission_group_manager.add_command("list-members", UserMemberPermissionGroupListMembers)
user_member_permission_group_manager.add_command("add-member", UserMemberPermissionGroupAddMember)
user_member_permission_group_manager.add_command("remove-member", UserMemberPermissionGroupRemoveMember)
permission_group_manager.add_command("user-member", user_member_permission_group_manager,
                                     help="User member permission group")


# MemberPermissionGroup
class MemberPermissionGroupCommand(PermissionGroupCommand):
    TYPE = "Member"

    @classmethod
    def get_permission_group(cls, name):
        permission_group = get_member_permission_group(name)
        if not permission_group:
            raise InvalidCommand("Member permission group %s does not exist" % name)
        return permission_group

    @classmethod
    def get_permission_groups(cls):
        return get_member_permission_groups()

    @classmethod
    def add_permission_group(cls, name, description):
        return add_member_permission_group(name, description)

    @classmethod
    def get_members(cls, name):
        return cls.get_permission_group(name).members

    @classmethod
    def format_member(cls, member):
        return "Member #%s" % member.id

    @classmethod
    def check_is_member(cls, group, member):
        if member not in group.members:
            raise InvalidCommand("Member id %s not in group %s" % (member.id, group.name))

    @classmethod
    def find_member(cls, _id):
        member = get_member(_id)
        if not member:
            raise InvalidCommand("User member id %s not found" % _id)
        return member


class MemberPermissionGroupRemovePermission(PermissionGroupRemovePermission, MemberPermissionGroupCommand):
    """
    Remove permission from member permission group
    """


class MemberPermissionGroupAddPermission(PermissionGroupAddPermission, MemberPermissionGroupCommand):
    """
    Add permission to member permission group
    """
    pass


class MemberPermissionGroupListPermissions(PermissionGroupListPermissions, MemberPermissionGroupCommand):
    """
    List member permission group permissions
    """
    pass


class ListMemberPermissionGroup(ListPermissionGroup, MemberPermissionGroupCommand):
    """
    List member permission group
    """


class RemoveMemberPermissionGroup(RemovePermissionGroup, MemberPermissionGroupCommand):
    """
    Remove member permission group
    """


class AddMemberPermissionGroup(AddPermissionGroup, MemberPermissionGroupCommand):
    """
    Add member permission group
    """


class MemberPermissionGroupListMembers(PermissionGroupListMembers, MemberPermissionGroupCommand):
    """
    List member permission group members
    """


class MemberPermissionGroupAddMember(PermissionGroupAddMember, MemberPermissionGroupCommand):
    """
    Add member to member permission group
    """


class MemberPermissionGroupRemoveMember(PermissionGroupRemoveMember, MemberPermissionGroupCommand):
    """
    Add member to member permission group
    """


member_permission_group_manager = Manager(usage="Manage member permission groups")
member_permission_group_manager.add_command("add", AddMemberPermissionGroup)
member_permission_group_manager.add_command("remove", RemoveMemberPermissionGroup)
member_permission_group_manager.add_command("list", ListMemberPermissionGroup)
member_permission_group_manager.add_command("list-permissions", MemberPermissionGroupListPermissions)
member_permission_group_manager.add_command('add-permission', MemberPermissionGroupAddPermission)
member_permission_group_manager.add_command('remove-permission', MemberPermissionGroupRemovePermission)
member_permission_group_manager.add_command("list-members", MemberPermissionGroupListMembers)
member_permission_group_manager.add_command("add-member", MemberPermissionGroupAddMember)
member_permission_group_manager.add_command("remove-member", MemberPermissionGroupRemoveMember)
permission_group_manager.add_command("member", member_permission_group_manager,
                                     help="Member permission group")
