from renki.core.lib.renkiapi import renkiapi
from renki.core.lib.exceptions import DoesNotExist
from renki.core.lib.auth.db import UserPermissionGroup, Permission, User
from renki.core.lib.utils import ok as ret_ok, conflict as ret_conflict, forbidden as ret_denied,\
    not_found as ret_not_found
from renki.core.lib.auth.basic_permissions import ViewUserPermissionGroups, EditUserPermissionGroups
from renki.core.lib.database.table import db
from flask import Blueprint

blueprint = Blueprint('user_permission_groups', __name__)


@renkiapi('GET', '/permission_groups/user',
          response={
              'type': 'object',
              'properties': {
                  'user_permission_groups': {
                      'type': 'array',
                      'items': {
                          'type': 'object'
                      }
                  }
              },
              'required': ['user_permission_groups'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_user_list(identity):
    """
    List all user permission groups and users and permissions associated with the groups
    :returns list of user permission groups in following format:
    user_permission_groups: {
        1: {
            name: group1,
            description: Group one is a group,
        }
    }
    """

    if not identity.has_permission(ViewUserPermissionGroups):
        return ret_denied('Not allowed')

    user_permission_groups = {}

    for user_permission_group in UserPermissionGroup.query.all():
        user_permission_groups[user_permission_group.id] = {
            'name': user_permission_group.name,
            'description': user_permission_group.description
        }

    return ret_ok({'user_permission_groups': user_permission_groups})


@renkiapi('POST', '/permission_groups/user',
          json={
              'type': 'object',
              'properties': {
                  'name': {'type': 'string'},
                  'description': {'type': 'string'}
              },
              'required': ['name', 'description'],
              'additionalProperties': False
          },
          response={
              'type': 'object',
              'properties': {
                  'permission_group_id': {'type': 'integer'}
              }
          }, blueprint=blueprint)
def permission_groups_user_add(identity, data):
    """
    Create new user permission group with name 'name' and description 'description'
    """
    if not identity.has_permission(EditUserPermissionGroups):
        return ret_denied('Not allowed')

    name = data['name']
    description = data['description']

    if UserPermissionGroup.query.filter(UserPermissionGroup.name == name).count():
        return ret_conflict('User permission group with name %s already exists' % name)

    permission_group = UserPermissionGroup()
    permission_group.name = name
    permission_group.description = description
    permission_group.save()

    db.session.commit()

    return ret_ok({'permission_group_id': permission_group.id})


@renkiapi('DELETE', '/permission_groups/user/<int:permission_group_id>',
          url_params={
              'type': 'object',
              'properties': {
                  'permission_group_id': {'type': 'integer'}
              },
              'required': ['permission_group_id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_user_delete(identity, data):
    """
    Delete user_permission_group with id 'permission_group_id'
    """

    if not identity.has_permission(EditUserPermissionGroups):
        return ret_denied('Not allowed')

    permission_group_id = int(data['permission_group_id'])

    try:
        permission_group = UserPermissionGroup.get(permission_group_id)
    except DoesNotExist:
        return ret_not_found('User permission group with id %i not found' % permission_group_id)

    db.session.delete(permission_group)
    db.session.commit()

    return ret_ok()


@renkiapi('PUT', '/permission_groups/user/<int:permission_group_id>',
          url_params={
              'type': 'object',
              'properties': {
                  'permission_group_id': {'type': 'integer'}
              },
              'required': ['permission_group_id'],
              'additionalProperties': False
          },
          json={
              'type': 'object',
              'properties': {
                  'name': {'type': 'string'},
                  'description': {'type': 'string'}
              },
              'required': ['name', 'description'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_user_update(identity, data):
    """
    Update name and description of user_permission_group with id 'permission_group_id'
    """
    if not identity.has_permission(EditUserPermissionGroups):
        return ret_denied('Not allowed')

    permission_group_id = int(data['permission_group_id'])

    try:
        permission_group = UserPermissionGroup.get(permission_group_id)
    except DoesNotExist:
        return ret_not_found('User permission group with id %i not found' % permission_group_id)

    name = data['name']
    description = data['description']

    permission_group.name = name
    permission_group.description = description
    permission_group.save()
    db.session.commit()

    return ret_ok()


@renkiapi('GET', '/permission_groups/user/<int:permission_group_id>/users',
          url_params={
              'type': 'object',
              'properties': {
                  'permission_group_id': {'type': 'integer'}
              },
              'required': ['permission_group_id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_user_users_list(identity, data):
    """
    List users in user_permission_group with id 'permission_group_id'
    """
    if not identity.has_permission(ViewUserPermissionGroups):
        return ret_denied('Not allowed')

    permission_group_id = int(data['permission_group_id'])

    try:
        permission_group = UserPermissionGroup.get(permission_group_id)
    except DoesNotExist:
        return ret_not_found('User permission group with id %i not found' % permission_group_id)

    users = {}

    for user in permission_group.users:
        users[user.id] = {
            'name': user.name,
            'auth_module': user.auth_module
        }

    return ret_ok({'users': users})


@renkiapi('POST', '/permission_groups/user/<int:permission_group_id>/users',
          url_params={
              'type': 'object',
              'properties': {
                  'permission_group_id': {'type': 'integer'}
              },
              'required': ['permission_group_id'],
              'additionalProperties': False
          },
          json={
              'user_id': {'type': str}
          }, blueprint=blueprint)
def permission_groups_user_users_add(identity, data):
    """
    Add user with id 'user_id' to user permission group with id 'permission_group_id'
    """
    if not identity.has_permission(EditUserPermissionGroups):
        return ret_denied('Not allowed')

    permission_group_id = int(data['permission_group_id'])

    try:
        permission_group = UserPermissionGroup.get(permission_group_id)
    except DoesNotExist:
        return ret_not_found('User permission group with id %i not found' % permission_group_id)

    user_id = data['user_id']

    try:
        user = User.get(user_id)
    except DoesNotExist:
        return ret_not_found('User with id %s not found' % user_id)

    permission_group.users.append(user)
    permission_group.save()
    db.session.commit()

    return ret_ok()


@renkiapi('DELETE', '/permission_groups/user/<int:permission_group_id>/users/<int:user_id>',
          url_params={
              'type': 'object',
              'properties': {
                  'permission_group_id': {'type': 'integer'},
                  'user_id': {'type': 'integer'}
              },
              'required': ['permission_group_id', 'user_id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_user_users_delete(identity, data):
    """
    Delete user with id 'user_id' from user permission group with id 'permission_group_id'
    """
    if not identity.has_permission(EditUserPermissionGroups):
        return ret_denied('Not allowed')

    permission_group_id = int(data['permission_group_id'])
    user_id = int(data['user_id'])

    try:
        permission_group = UserPermissionGroup.get(permission_group_id)
    except DoesNotExist:
        return ret_not_found('User permission group with id %i not found' % permission_group_id)

    try:
        user = User.get(user_id)
    except DoesNotExist:
        return ret_not_found('User with id %s not found' % user_id)

    permission_group.users.remove(user)
    db.session.commit()

    return ret_ok()


@renkiapi('GET', '/permission_groups/user/<int:permission_group_id>/permissions',
          url_params={
              'type': 'object',
              'properties': {
                  'permission_group_id': {'type': 'integer'}
              },
              'required': ['permission_group_id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_user_permissions_list(identity, data):
    """
    List permissions in user_permission_group with id 'permission_group_id'
    """
    if not identity.has_permission(ViewUserPermissionGroups):
        return ret_denied('Not allowed')

    permission_group_id = int(data['permission_group_id'])

    try:
        permission_group = UserPermissionGroup.get(permission_group_id)
    except DoesNotExist:
        return ret_not_found('User permission group with id %i not found' % permission_group_id)

    permissions = {}

    for permission in permission_group.permissions:
        permissions[permission.id] = {
            'name': permission.name,
            'description': permission.description,
            'service_id': permission.service_id,
            'is_global': permission.is_global
        }

    return ret_ok({'permissions': permissions})


@renkiapi('POST', '/permission_groups/user/<int:permission_group_id>/permissions',
          url_params={
              'type': 'object',
              'properties': {
                  'permission_group_id': {'type': 'integer'}
              },
              'required': ['permission_group_id'],
              'additionalProperties': False
          },
          json={
              'type': 'object',
              'properties': {
                  'permission_id': {'type': 'integer'}
              },
              'required': ['permission_id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_user_permissions_add(identity, data):
    """
    Add permission with id permission_id to user permission group with id permission_group_id
    """
    if not identity.has_permission(EditUserPermissionGroups):
        return ret_denied('Not allowed')

    permission_group_id = int(data['permission_group_id'])
    permission_id = data['permission_id']

    try:
        permission_group = UserPermissionGroup.get(permission_group_id)
    except DoesNotExist:
        return ret_not_found('User permission group with id %i not found' % permission_group_id)

    try:
        permission = Permission.get(permission_id)
    except DoesNotExist:
        return ret_not_found('Permission with id %i not found' % permission_id)

    permission_group.permissions.append(permission)
    permission_group.save()
    db.session.commit()

    return ret_ok()


@renkiapi('DELETE', '/permission_groups/user/<int:permission_group_id>/permissions/<int:permission_id>',
          url_params={
              'type': 'object',
              'properties': {
                  'permission_group_id': {'type': 'integer'},
                  'permission_id': {'type': 'integer'}
              },
              'required': ['permission_group_id', 'permission_id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_user_permissions_delete(identity, data):
    """
    Delete permission with id 'permission_id' from user_permission_group with id 'permission_group_id'
    """
    if not identity.has_permission(EditUserPermissionGroups):
        return ret_denied('Not allowed')

    permission_group_id = int(data['permission_group_id'])
    permission_id = int(data['permission_id'])

    try:
        permission_group = UserPermissionGroup.get(permission_group_id)
    except DoesNotExist:
        return ret_not_found('User permission group with id %i not found' % permission_group_id)

    try:
        permission = Permission.get(permission_id)
    except DoesNotExist:
        return ret_not_found('Permission with id %i not found' % permission_id)

    permission_group.permissions.remove(permission)
    permission_group.save()
    db.session.commit()

    return ret_ok()
