from renki.core.lib.renkiapi import renkiapi
from renki.core.lib.utils import ok, forbidden
from renki.core.lib.database.basic_tables import Service
from renki.core.lib.auth.basic_permissions import ViewServices
from flask import Blueprint

blueprint = Blueprint('services', __name__)


@renkiapi('GET', '/services',
          response={
              'type': 'array',
              'items': {
                  'type': 'object',
                  'properties': {
                      'id': {'type': 'integer'},
                      'name': {'type': 'string'},
                      'type': {'type': 'string'}
                  }
              }
          }, blueprint=blueprint)
def services_list(identity):
    """
    List all available services

    :param identity: User identity
    """

    services = []

    service_ids = []
    for permission in identity.get_permissions_unique_to_service():
        if not permission.is_global:
            service_ids.append(permission.service_id)

    for service in Service.query.filter(Service.id.in_(service_ids)):
        services.append({
            'id': service.id,
            'name': service.name,
            'service_type': service.service_type
        })

    return ok({'services': services})


@renkiapi('GET', '/services/<int:id>',
          url_params={
              'type': 'object',
              'properties': {
                  'id': {'type': 'integer'}
              }
          },
          response={
              'type': 'object',
              'properties': {
                  'id': {'type': 'integer'},
                  'name': {'type': 'string'},
                  'type': {'type': 'string'},
                  'servers': {
                      'type': 'array',
                      'items': {
                          'type': 'object',
                          'properties': {
                              'id': {'type': 'integer'},
                              'name': {'type': 'string'}
                          }
                      }
                  }
              }
          }, blueprint=blueprint)
def services_get_service(identity, data):
    """
    List servers associated with service

    :param identity: User identity
    :param data: Data associated with the request
    """
    if not identity.has_permission(ViewServices):
        return forbidden()

    service_id = int(data['id'])

    service = Service.get(service_id)

    servers = []
    for server in service.servers:
        servers.append({
            'id': server.id,
            'name': server.name
        })

    return ok({
        'id': service.id,
        'name': service.name,
        'type': service.service_type,
        'servers': servers
    })
