from renki.core.lib.exceptions import Invalid, DoesNotExist, DatabaseError
from sqlalchemy import Column, Integer, String, DateTime
from sqlalchemy.orm.exc import NoResultFound
from sqlalchemy.exc import SQLAlchemyError
from flask.ext.sqlalchemy import SQLAlchemy
from flask_migrate import Migrate
from datetime import datetime

import logging
logger = logging.getLogger('dbconnection')

db = SQLAlchemy()
migrate = Migrate()


class RenkiTable(db.Model):
    __abstract__ = True

    id = Column("id", Integer, primary_key=True, nullable=False)

    @classmethod
    def get(cls, id_):
        if id_:
            try:
                id_ = int(id_)
            except ValueError:
                logger.error("Get with invalid database id %s" % id_)
                raise Invalid('ID must be integer')
            try:
                c = db.session.query(cls).filter(cls.id == id_).one()
            except NoResultFound:
                raise DoesNotExist('Object with id %d does not exist' % id_)
            except SQLAlchemyError as e:
                logger.exception(e)
                raise DatabaseError('Cannot get object with id %d' % id_)
            return c
        raise Invalid('ID must be integer')

    def validate(self):
        """
        Validate this object
        @returns true if object is valid
        @raises Invalid if object is invalid
        """
        raise Invalid("Dummy validator")

    def delete(self):
        """
        Delete this object from database
        """
        db.session.delete(self)
        return True

    def save(self, commit=False):
        """
        Save this object to database by updating existing row or inserting
        new one.
        """
        self.validate()
        db.session.add(self)
        if commit is True:
            db.session.save_commit()
        return True

    def as_dict(self):
        """
        Return this object columns as dict object
        """
        ret = {}
        for i in self.__table__.columns.keys():
            if getattr(self, i).__class__.__name__ in ['datetime']:
                ret[i] = str(getattr(self, i))
            elif i not in ['deleted']:
                ret[i] = getattr(self, i)
        return ret

    # Variation of as_dict mainly to work with automatic history tables which don't have __table__
    def to_dict(self):
        """
        Return this object columns as dict object
        """
        ret = {}
        for i in vars(self):
            if i[0] != '_':
                if getattr(self, i).__class__.__name__ in ['datetime']:
                    ret[i] = str(getattr(self, i))
                elif i not in ['deleted']:
                    ret[i] = getattr(self, i)
        return ret


class RenkiDataTable(RenkiTable):
    __abstract__ = True
    comment = Column("comment", String, nullable=False, default='')
    deleted = Column("deleted", Integer, nullable=True, default=None)
    timestamp = Column("timestamp", DateTime, nullable=False, default=datetime.now)


class RenkiUserTable(RenkiDataTable):
    __abstract__ = True
    user_id = Column('user_id', Integer, nullable=False)
