from renki.core.lib.renkiapi import renkiapi
from renki.core.lib.exceptions import DoesNotExist
from renki.core.lib.auth.db import MemberPermissionGroup, Permission, Member
from renki.core.lib.utils import ok as ret_ok, conflict as ret_conflict, forbidden as ret_denied,\
    not_found as ret_not_found
from renki.core.lib.auth.basic_permissions import ViewMemberPermissionGroups, EditMemberPermissionGroups
from renki.core.lib.database.table import db
from flask import Blueprint

blueprint = Blueprint('member_permission_group', __name__)


@renkiapi('GET', '/permission_groups/member',
          response={
              'type': 'array'
          }, blueprint=blueprint)
def permission_groups_member_list(identity):
    """
    List all member permission groups
    :returns list of member permission groups in following format:
    member_permission_groups: {
        1: {
            name: group1,
            description: Group one is a group,
        }
    }
    """
    if not identity.has_permission(ViewMemberPermissionGroups):
        return ret_denied('Not allowed')

    member_permission_groups = []

    for member_permission_group in MemberPermissionGroup.query.all():
        member_permission_groups.append({
            'id': member_permission_group.id,
            'name': member_permission_group.name,
            'description': member_permission_group.description
        })

    return ret_ok(member_permission_groups)


@renkiapi('POST', '/permission_groups/member',
          json={
              'type': 'object',
              'properties': {
                  'name': {'type': 'string'},
                  'description': {'type': 'string'}
              },
              'required': ['name', 'description'],
              'additionalProperties': False
          },
          response={
              'type': 'object',
              'properties': {
                  'id': {'type': 'integer'},
                  'name': {'type': 'string'},
                  'description': {'type': 'string'}
              },
              'required': ['id', 'name', 'description'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_member_add(identity, data):
    """
    Create new member permission group with name 'name' and description 'description'
    """
    if not identity.has_permission(EditMemberPermissionGroups):
        return ret_denied('Not allowed')

    name = data['name']
    description = data['description']

    if MemberPermissionGroup.query.filter(MemberPermissionGroup.name == name).count():
        return ret_conflict('Member permission group with name %s already exists' % name)

    permission_group = MemberPermissionGroup()
    permission_group.name = name
    permission_group.description = description
    permission_group.save()

    db.session.commit()

    return ret_ok({
        'id': permission_group.id,
        'name': permission_group.name,
        'description': permission_group.description
    })


@renkiapi('DELETE', '/permission_groups/member/<int:id>',
          url_params={
              'type': 'object',
              'properties': {
                  'id': {'type': 'integer'}
              },
              'required': ['id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_member_delete(identity, data):
    """
    Delete member_permission_group with id 'permission_group_id'
    """
    if not identity.has_permission(EditMemberPermissionGroups):
        return ret_denied('Not allowed')

    permission_group_id = int(data['id'])

    try:
        permission_group = MemberPermissionGroup.get(permission_group_id)
    except DoesNotExist:
        return ret_not_found('Member permission group with id %i not found' % permission_group_id)

    db.session.delete(permission_group)
    db.session.commit()

    return ret_ok({})


@renkiapi('PUT', '/permission_groups/member/<int:id>',
          url_params={
              'type': 'object',
              'properties': {
                  'id': {'type': 'integer'}
              },
              'required': ['id'],
              'additionalProperties': False
          },
          json={
              'type': 'object',
              'properties': {
                  'id': {'type': 'integer'},
                  'name': {'type': 'string'},
                  'description': {'type': 'string'}
              },
              'required': ['name', 'description'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_member_update(identity, data):
    """
    Update name and description of member_permission_group with id 'permission_group_id'
    """
    if not identity.has_permission(EditMemberPermissionGroups):
        return ret_denied('Not allowed')

    permission_group_id = int(data['id'])

    try:
        permission_group = MemberPermissionGroup.get(permission_group_id)
    except DoesNotExist:
        return ret_not_found('Member permission group with id %i not found' % permission_group_id)

    name = data['name']
    description = data['description']

    permission_group.name = name
    permission_group.description = description
    permission_group.save()
    db.session.commit()

    return ret_ok({
        'id': permission_group.id,
        'name': permission_group.name,
        'description': permission_group.description
    })


@renkiapi('GET', '/permission_groups/member/<int:id>/members',
          url_params={
              'type': 'object',
              'properties': {
                  'id': {'type': 'integer'}
              },
              'required': ['id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_member_members_list(identity, data):
    """
    List members in member_permission_group with id 'permission_group_id'
    """
    if not identity.has_permission(ViewMemberPermissionGroups):
        return ret_denied('Not allowed')

    permission_group_id = int(data['id'])

    try:
        permission_group = MemberPermissionGroup.get(permission_group_id)
    except DoesNotExist:
        return ret_not_found('Member permission group with id %i not found' % permission_group_id)

    members = []

    for member in permission_group.members:
        members.append(member.id)

    return ret_ok({'members': members})


@renkiapi('POST', '/permission_groups/member/<int:permission_group_id>/members',
          url_params={
              'type': 'object',
              'properties': {
                  'permission_group_id': {'type': 'integer'}
              },
              'required': ['permission_group_id'],
              'additionalProperties': False
          },
          json={
              'type': 'object',
              'properties': {
                  'member_id': {'type': 'integer'}
              },
              'required': ['member_id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_member_members_add(identity, data):
    """
    Add member with id 'member_id' to member permission group with id 'permission_group_id'
    """
    if not identity.has_permission(EditMemberPermissionGroups):
        return ret_denied('Not allowed')

    permission_group_id = int(data['permission_group_id'])

    try:
        permission_group = MemberPermissionGroup.get(permission_group_id)
    except DoesNotExist:
        return ret_not_found('Member permission group with id %i not found' % permission_group_id)

    member_id = int(data['member_id'])

    try:
        member = Member.get(member_id)
    except DoesNotExist:
        return ret_not_found('Member with id %s not found' % member_id)

    permission_group.members.append(member)
    permission_group.save()
    db.session.commit()

    return ret_ok()


@renkiapi('DELETE', '/permission_groups/member/<int:permission_group_id>/members/<int:member_id>',
          url_params={
              'type': 'object',
              'properties': {
                  'permission_group_id': {'type': 'integer'},
                  'member_id': {'type': 'integer'}
              },
              'required': ['permission_group_id', 'member_id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_member_members_delete(identity, data):
    """
    Delete member with id 'member_id' from member permission group with id 'permission_group_id'
    """
    if not identity.has_permission(EditMemberPermissionGroups):
        return ret_denied('Not allowed')

    permission_group_id = int(data['permission_group_id'])
    member_id = int(data['member_id'])

    try:
        permission_group = MemberPermissionGroup.get(permission_group_id)
    except DoesNotExist:
        return ret_not_found('Member permission group with id %i not found' % permission_group_id)

    try:
        member = Member.get(member_id)
    except DoesNotExist:
        return ret_not_found('Member with id %s not found' % member_id)

    permission_group.members.remove(member)
    db.session.commit()

    return ret_ok()


@renkiapi('GET', '/permission_groups/member/<int:permission_group_id>/permissions',
          url_params={
              'type': 'object',
              'properties': {
                  'permission_group_id': {'type': 'integer'}
              },
              'required': ['permission_group_id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_member_permissions_list(identity, data):
    """
    List permissions in member_permission_group with id 'permission_group_id'
    """
    if not identity.has_permission(ViewMemberPermissionGroups):
        return ret_denied('Not allowed')

    permission_group_id = int(data['permission_group_id'])

    try:
        permission_group = MemberPermissionGroup.get(permission_group_id)
    except DoesNotExist:
        return ret_not_found('Member permission group with id %i not found' % permission_group_id)

    permissions = {}

    for permission in permission_group.permissions:
        permissions[permission.id] = {
            'name': permission.name,
            'description': permission.description,
            'service_id': permission.service_id,
            'is_global': permission.is_global
        }

    return ret_ok({'permissions': permissions})


@renkiapi('POST', '/permission_groups/member/<int:permission_group_id>/permissions',
          url_params={
              'type': 'object',
              'properties': {
                  'permission_group_id': {'type': 'integer'}
              },
              'required': ['permission_group_id'],
              'additionalProperties': False
          },
          json={
              'type': 'object',
              'properties': {
                  'permission_id': {'type': 'integer'}
              },
              'required': ['permission_id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_member_permissions_add(identity, data):
    """
    Add permission with id permission_id to member permission group with id permission_group_id
    """
    if not identity.has_permission(EditMemberPermissionGroups):
        return ret_denied('Not allowed')

    permission_group_id = int(data['permission_group_id'])

    try:
        permission_group = MemberPermissionGroup.get(permission_group_id)
    except DoesNotExist:
        return ret_not_found('Member permission group with id %i not found' % permission_group_id)

    permission_id = data['permission_id']

    try:
        permission = Permission.get(permission_id)
    except DoesNotExist:
        return ret_not_found('Permission with id %i not found' % permission_id)

    permission_group.permissions.append(permission)
    permission_group.save()
    db.session.commit()

    return ret_ok()


@renkiapi('DELETE', '/permission_groups/member/<int:permission_group_id>/permissions/<int:permission_id>',
          url_params={
              'type': 'object',
              'properties': {
                  'permission_group_id': {'type': 'integer'},
                  'permission_id': {'type': 'integer'}
              },
              'required': ['permission_group_id', 'permission_id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_member_permissions_delete(identity, data):
    """
    Delete permission with id 'permission_id' from member_permission_group with id 'permission_group_id'
    """
    if not identity.has_permission(EditMemberPermissionGroups):
        return ret_denied('Not allowed')

    permission_group_id = int(data['permission_group_id'])
    permission_id = int(data['permission_id'])

    try:
        permission_group = MemberPermissionGroup.get(permission_group_id)
    except DoesNotExist:
        return ret_not_found('Member permission group with id %i not found' % permission_group_id)

    try:
        permission = Permission.get(permission_id)
    except DoesNotExist:
        return ret_not_found('Permission with id %i not found' % permission_id)

    permission_group.permissions.remove(permission)
    permission_group.save()
    db.session.commit()

    return ret_ok()
