from renki.core.lib.renkiapi import renkiapi
from renki.core.lib.auth.db import UserToMember
from renki.core.lib.utils import ok, forbidden
from renki.core.lib.database.table import db
from flask import Blueprint

blueprint = Blueprint('session', __name__)


@renkiapi('GET', '/session/info',
          response={
              'type': 'object',
              'properties': {
                  'authToken': {'type': 'string'},
                  'memberId': {'type': 'integer'}
              }
          }, blueprint=blueprint)
def session_info(identity):
    """
    Show info about current session

    :param identity: User identity
    """
    ret = {'authToken': identity.token}

    if identity.member is not None:
        ret.update({'memberId': identity.member.id})

    return ok(ret)


@renkiapi('GET', '/session/members',
          response={
              'type': 'array',
              'items': {
                  'type': 'integer'
              }
          }, blueprint=blueprint)
def session_list_members(identity):
    """
    List members which user can act as

    :param identity: User identity
    """
    query = UserToMember.query.filter(UserToMember.user_id == identity.user.user_id)

    members = []
    for user_to_member in query.all():
        members.append(user_to_member.member_id)

    return ok(members)


@renkiapi('POST', '/session/members',
          json={
              'type': 'object',
              'properties': {
                  'id': {'type': 'integer'}
              }
          }, blueprint=blueprint)
def session_select_member(identity, data):
    """
    Select member whose info user is accessing

    :param identity: User identity
    :param data: Data associated with the request
    """
    member_id = int(data['id'])

    query = UserToMember.query.filter(UserToMember.user_id == identity.user.user_id) \
        .filter(UserToMember.member_id == member_id)

    if query.count() == 1:
        identity.member_id = query.one().member_id
    else:
        return forbidden('Not allowed to act as member %i' % member_id)

    db.session.commit()

    return ok()


@renkiapi('GET', '/session/permissions',
          response={
              'type': 'object',
              'properties': {
                  'globalPermissions': {
                      'type': 'array'
                  },
                  'modulePermissions': {
                      'type': 'array'
                  }
              },
              'additionalProperties': False,
              'required': ['globalPermissions', 'modulePermissions']
          }, blueprint=blueprint)
def session_permissions(identity):
    """
    List permissions of user taking into account current member that user is acting as

    :param identity: User identity
    """
    global_permissions = []
    module_permissions = []

    # Available permissions come from the currently used _identity_ not user
    available_permissions = identity.get_permissions()

    for permission in available_permissions:
        if permission.is_global:
            global_permissions.append({
                'id': permission.id,
                'name': permission.name
            })
        else:
            module_permissions.append({
                'id': permission.id,
                'name': permission.name,
                'serviceId': permission.service_id
            })

    return ok({'globalPermissions': global_permissions, 'modulePermissions': module_permissions})
