from renki.core.lib.renkiapi import renkiapi
from renki.core.lib.exceptions import DoesNotExist
from renki.core.lib.auth.db import AuthTokens, Member
from renki.core.lib.utils import ok as ret_ok, unauthorized as ret_noauth, conflict as ret_conflict,\
    forbidden as ret_denied, not_found as ret_not_found
from renki.core.lib.auth.basic_permissions import ViewMembers, EditMembers
from renki.core.lib.database.table import db
from flask import Blueprint

blueprint = Blueprint('members', __name__)


@renkiapi('GET', '/members',
          response={
              'type': 'array'
          }, blueprint=blueprint)
def members_list(identity):
    if not identity.has_permission(ViewMembers):
        return ret_denied('Not allowed')

    members = []
    for member in Member.query.all():
        members.append(member.id)

    return ret_ok(members)


@renkiapi('POST', '/members',
          json={
              'type': 'object',
              'properties': {
                  'id': {'type': 'integer'}
              },
              'required': ['id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def members_add(identity, data):
    if not identity.has_permission(EditMembers):
        return ret_denied('Not allowed')

    member_id = data['id']

    if Member.query.filter(Member.id == member_id).count():
        return ret_conflict('Member with id %i already exists' % id)

    member = Member()
    member.id = member_id
    member.save()

    db.session.commit()

    return ret_ok()


@renkiapi('DELETE', '/members/<int:id>',
          url_params={
              'type': 'object',
              'properties': {
                  'id': {'type': 'integer'}
              },
              'required': ['id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def members_delete(auth_token, member_id):
    try:
        identity = AuthTokens.get_token(auth_token)
    except DoesNotExist:
        return ret_noauth('Not authenticated')

    if not identity.has_permission(EditMembers):
        return ret_denied('Not allowed')

    try:
        member = Member.get(member_id)
    except DoesNotExist:
        return ret_not_found('Member with id %i not found' % member_id)

    db.session.delete(member)
    db.session.commit()

    return ret_ok()
