from renki.core.lib.test_utils import BasicTest
from renki.core.lib.auth.authentication_modules.basic import BasicAuthenticationModule
from renki.core.lib.auth.db import Permission, Member, UserPermissionGroup, UserMemberPermissionGroup, UserToMember
from renki.core.lib.auth.basic_permissions import ViewUserMemberPermissionGroups, EditUserMemberPermissionGroups,\
    ViewPersonalData
from renki.core.lib.database.table import db
import unittest
import jsonschema


class UserMemberPermissionGroupTest(BasicTest):
    def setUp(self):
        super(UserMemberPermissionGroupTest, self).setUp()

        self._view_permission_groups = Permission()
        self._view_permission_groups.name = ViewUserMemberPermissionGroups.name
        self._view_permission_groups.is_global = True
        self._view_permission_groups.save()

        self._edit_permission_groups = Permission()
        self._edit_permission_groups.name = EditUserMemberPermissionGroups.name
        self._edit_permission_groups.is_global = True
        self._edit_permission_groups.save()

        self._dummy_permission = Permission()
        self._dummy_permission.name = ViewPersonalData.name
        self._dummy_permission.is_global = True
        self._dummy_permission.save()

        db.session.commit()

        self._basic_user_1 = BasicAuthenticationModule.register_user('basic1', 'basic1')
        self._basic_user_2 = BasicAuthenticationModule.register_user('basic2', 'basic2')
        self._dummy_user = BasicAuthenticationModule.register_user('dummy', 'dummy')
        self._dummy_member = Member()
        self._dummy_member.save()
        db.session.commit()

        self._dummy_user_to_dummy_member = UserToMember()
        self._dummy_user_to_dummy_member.user_id = self._dummy_user.id
        self._dummy_user_to_dummy_member.member_id = self._dummy_member.id
        self._dummy_user_to_dummy_member.save()

        self._admin_permission_group = UserPermissionGroup()
        self._admin_permission_group.name = 'Admin'
        self._admin_permission_group.description = 'Admins'
        self._admin_permission_group.add_permission(self._edit_permission_groups)
        self._admin_permission_group.add_permission(self._view_permission_groups)
        self._admin_permission_group.add_user(self._basic_user_1)
        self._admin_permission_group.save()

        self._basic_user_member_permission_group = UserMemberPermissionGroup()
        self._basic_user_member_permission_group.name = 'Basic'
        self._basic_user_member_permission_group.description = 'Group connecting users to their own members'
        self._basic_user_member_permission_group.add_permission(self._edit_permission_groups)
        self._basic_user_member_permission_group.add_permission(self._view_permission_groups)
        self._basic_user_member_permission_group.save()
        db.session.commit()


class TestListUserMemberPermissionGroupsRoute(UserMemberPermissionGroupTest):
    _user_permission_groups_schema = {
        'user_member_permission_groups': {
            'type': 'object',
            'patternProperties': {
                '^[0-9]+$': {
                    'type': 'object',
                    'properties': {
                        'name': {
                            'type': 'string'
                        },
                        'description': {
                            'type': 'string'
                        }
                    },
                    'required': ['name', 'description'],
                    'additionalProperties': False
                }
            }
        },
        'additionalProperties': True
    }

    def setUp(self):
        super(TestListUserMemberPermissionGroupsRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.get('/permission_groups/user_member', {}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')

        assert self.get('/permission_groups/user_member', {}, expect_errors=True).status_int == 403

    def test_valid(self):
        self.auth('basic1', 'basic1')

        data = self.get('/permission_groups/user_member', {}).json

        jsonschema.validate(data, self._user_permission_groups_schema)

        basic_permission_group = data['user_member_permission_groups'][str(self._basic_user_member_permission_group.id)]
        assert basic_permission_group['name'] == self._basic_user_member_permission_group.name
        assert basic_permission_group['description'] == self._basic_user_member_permission_group.description


class TestAddUserMemberPermissionGroupRoute(UserMemberPermissionGroupTest):
    def setUp(self):
        super(TestAddUserMemberPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.post('/permission_groups/user_member',
                         {'name': 'test_group',
                          'description': 'test group is a test group'}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')

        assert self.post('/permission_groups/user_member',
                         {'name': 'test_group',
                          'description': 'test group is a test group'}, expect_errors=True).status_int == 403

    def test_duplicate(self):
        self.auth('basic1', 'basic1')

        permission_group = UserMemberPermissionGroup()
        permission_group.name = 'test_group'
        permission_group.description = 'test group is a test group'
        permission_group.save()
        db.session.commit()

        assert self.post('/permission_groups/user_member',
                         {'name': 'test_group',
                          'description': 'test group is a test group'}, expect_errors=True).status_int == 409

    def test_valid(self):
        self.auth('basic1', 'basic1')

        assert UserMemberPermissionGroup.query.filter(UserMemberPermissionGroup.name == 'test_group').count() == 0

        self.post('/permission_groups/user_member', {'name': 'test_group', 'description': 'test group is a test group'})

        assert UserMemberPermissionGroup.query.filter(UserMemberPermissionGroup.name == 'test_group').count() == 1


class TestUpdateUserMemberPermissionGroupRoute(UserMemberPermissionGroupTest):
    def setUp(self):
        super(TestUpdateUserMemberPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.put('/permission_groups/user_member/1', {}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')

        assert self.put('/permission_groups/user_member/1',
                        {'name': 'test_group',
                         'description': 'test group is a test group'}, expect_errors=True).status_int == 403

    def test_non_existent_group(self):
        self.auth('basic1', 'basic1')

        assert self.put('/permission_groups/user_member/123',
                        {'name': 'test_group',
                         'description': 'test group is a test group'}, expect_errors=True).status_int == 404

    def test_valid(self):
        self.auth('basic1', 'basic1')

        assert UserMemberPermissionGroup.get(
            self._basic_user_member_permission_group.id).name == self._basic_user_member_permission_group.name

        self.put('/permission_groups/user_member/%i' % self._basic_user_member_permission_group.id,
                 {'name': 'test_group', 'description': 'test group is a test group'})

        assert UserMemberPermissionGroup.get(self._basic_user_member_permission_group.id).name == 'test_group'


class TestDeleteUserMemberPermissionGroupRoute(UserMemberPermissionGroupTest):
    def setUp(self):
        super(TestDeleteUserMemberPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.delete('/permission_groups/user_member/1', {}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')
        assert self.delete('/permission_groups/user_member/1', {}, expect_errors=True).status_int == 403

    def test_non_existent_group(self):
        self.auth('basic1', 'basic1')
        assert self.delete('/permission_groups/user_member/123', {}, expect_errors=True).status_int == 404

    def test_valid(self):
        self.auth('basic1', 'basic1')
        assert UserMemberPermissionGroup.query.filter(
            UserMemberPermissionGroup.id == self._admin_permission_group.id).count()
        self.delete('/permission_groups/user_member/%i' % self._admin_permission_group.id, {})
        assert not UserMemberPermissionGroup.query.filter(
            UserMemberPermissionGroup.id == self._admin_permission_group.id).count()


class TestListUsersInUserMemberPermissionGroupRoute(UserMemberPermissionGroupTest):
    _user_members_schema = {
        'user_members': {
            'type': 'array',
            'items': {
                'type': 'array',
                'items': {
                    'type': 'integer'
                }
            }
        },
        'additionalProperties': True
    }

    def setUp(self):
        super(TestListUsersInUserMemberPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.get('/permission_groups/user_member/1/user_members', {}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')
        assert self.get('/permission_groups/user_member/1/user_members', {}, expect_errors=True).status_int == 403

    def test_non_existent_group(self):
        self.auth('basic1', 'basic1')
        assert self.get('/permission_groups/user_member/123/user_members', {}, expect_errors=True).status_int == 404

    def test_valid(self):
        self.auth('basic1', 'basic1')
        self._basic_user_member_permission_group.user_members.append(self._dummy_user_to_dummy_member)
        self._basic_user_member_permission_group.save()
        db.session.commit()
        data = self.get('/permission_groups/user_member/%i/user_members' % self._basic_user_member_permission_group.id,
                        {}).json
        jsonschema.validate(data, self._user_members_schema)

        assert data['user_members'][0][0] == self._dummy_user.id
        assert data['user_members'][0][1] == self._dummy_member.id


class TestAddUserToUserMemberPermissionGroupRoute(UserMemberPermissionGroupTest):
    def setUp(self):
        super(TestAddUserToUserMemberPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.post('/permission_groups/user_member/1/user_members',
                         {'user_id': self._dummy_user.id}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')
        assert self.post('/permission_groups/user_member/1/user_members',
                         {'user_id': self._dummy_user.id,
                          'member_id': self._dummy_member.id}, expect_errors=True).status_int == 403

    def test_non_existent_group(self):
        self.auth('basic1', 'basic1')
        assert self.post('/permission_groups/user_member/123/user_members',
                         {'user_id': self._dummy_user.id,
                          'member_id': self._dummy_member.id}, expect_errors=True).status_int == 404

    def test_non_existent_user(self):
        self.auth('basic1', 'basic1')
        assert self.post('/permission_groups/user_member/1/user_members',
                         {'user_id': 123,
                          'member_id': self._dummy_member.id}, expect_errors=True).status_int == 404

    def test_non_existent_member(self):
        self.auth('basic1', 'basic1')
        assert self.post('/permission_groups/user_member/1/user_members',
                         {'user_id': self._dummy_user.id,
                          'member_id': 123}, expect_errors=True).status_int == 404

    def test_valid(self):
        self.auth('basic1', 'basic1')
        assert len(self._basic_user_member_permission_group.user_members) == 0
        self.post('/permission_groups/user_member/%i/user_members' % self._basic_user_member_permission_group.id,
                  {'user_id': self._dummy_user.id,
                   'member_id': self._dummy_member.id})
        assert len(self._basic_user_member_permission_group.user_members) == 1


class TestDeleteUserFromUserMemberPermissionGroupRoute(UserMemberPermissionGroupTest):
    def setUp(self):
        super(TestDeleteUserFromUserMemberPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.delete('/permission_groups/user_member/1/user_members/123/123', {},
                           expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')
        assert self.delete('/permission_groups/user_member/1/user_members/123/123', {},
                           expect_errors=True).status_int == 403

    def test_non_existent_group(self):
        self.auth('basic1', 'basic1')
        assert self.delete('/permission_groups/user_member/123/user_members/%i/%i' % (self._dummy_user.id,
                                                                                      self._dummy_member.id), {},
                           expect_errors=True).status_int == 404

    def test_non_existent_user(self):
        self.auth('basic1', 'basic1')
        assert self.delete('/permission_groups/user_member/1/user_members/123/%i' % self._dummy_member.id, {},
                           expect_errors=True).status_int == 404

    def test_non_existent_member(self):
        self.auth('basic1', 'basic1')
        assert self.delete('/permission_groups/user_member/1/user_members/%i/123' % self._dummy_user.id, {},
                           expect_errors=True).status_int == 404

    def test_valid(self):
        self.auth('basic1', 'basic1')
        self._basic_user_member_permission_group.user_members.append(self._dummy_user_to_dummy_member)
        self._basic_user_member_permission_group.save()
        db.session.commit()

        assert self._dummy_user_to_dummy_member in self._basic_user_member_permission_group.user_members
        self.delete('/permission_groups/user_member/%i/user_members/%i/%i'
                    % (self._basic_user_member_permission_group.id,
                       self._dummy_user.id,
                       self._dummy_member.id), {})
        assert self._dummy_user_to_dummy_member not in self._basic_user_member_permission_group.user_members


class TestListPermissionsInUserMemberPermissionGroupRoute(UserMemberPermissionGroupTest):
    _permissions_schema = {
        'permissions': {
            'type': 'object',
            'patternProperties': {
                '^[0-9]+$': {
                    'type': 'object',
                    'properties': {
                        'name': {
                            'type': 'string'
                        },
                        'description': {
                            'type': 'string'
                        },
                        'service_id': {
                            'type': 'integer'
                        },
                        'is_global': {
                            'type': 'boolean'
                        }
                    },
                    'required': ['name', 'description', 'service_id', 'is_global'],
                    'additionalProperties': False
                }
            }
        },
        'additionalProperties': True
    }

    def setUp(self):
        super(TestListPermissionsInUserMemberPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.get('/permission_groups/user_member/1/permissions', {}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')
        assert self.get('/permission_groups/user_member/1/permissions', {}, expect_errors=True).status_int == 403

    def test_non_existent_group(self):
        self.auth('basic1', 'basic1')
        assert self.get('/permission_groups/user_member/123/permissions', {}, expect_errors=True).status_int == 404

    def test_valid(self):
        self.auth('basic1', 'basic1')
        data = self.get('/permission_groups/user_member/%i/permissions' % self._admin_permission_group.id, {}).json
        jsonschema.validate(data, self._permissions_schema)
        permission = data['permissions'][str(self._edit_permission_groups.id)]
        assert permission['name'] == self._edit_permission_groups.name


class TestAddPermissionToUserMemberPermissionGroupRoute(UserMemberPermissionGroupTest):
    def setUp(self):
        super(TestAddPermissionToUserMemberPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.post('/permission_groups/user_member/1/permissions',
                         {'permission_id': self._dummy_permission.id}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')
        assert self.post('/permission_groups/user_member/1/permissions',
                         {'permission_id': self._dummy_permission.id}, expect_errors=True).status_int == 403

    def test_non_existent_group(self):
        self.auth('basic1', 'basic1')
        assert self.post('/permission_groups/user_member/123/permissions',
                         {'permission_id': self._dummy_permission.id}, expect_errors=True).status_int == 404

    def test_non_existent_permission(self):
        self.auth('basic1', 'basic1')
        assert self.post('/permission_groups/user_member/1/permissions',
                         {'permission_id': 123}, {}, expect_errors=True).status_int == 404

    def test_valid(self):
        self.auth('basic1', 'basic1')
        assert self._dummy_permission not in self._basic_user_member_permission_group.permissions
        self.post('/permission_groups/user_member/%i/permissions' % self._basic_user_member_permission_group.id,
                  {'permission_id': self._dummy_permission.id})
        assert self._dummy_permission in self._basic_user_member_permission_group.permissions


class TestDeletePermissionFromUserMemberPermissionGroupRoute(UserMemberPermissionGroupTest):
    def setUp(self):
        super(TestDeletePermissionFromUserMemberPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.delete('/permission_groups/user_member/1/permissions/123', {}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')
        assert self.delete('/permission_groups/user_member/1/permissions/123', {}, expect_errors=True).status_int == 403

    def test_non_existent_group(self):
        self.auth('basic1', 'basic1')
        assert self.delete('/permission_groups/user_member/123/permissions/1', {}, expect_errors=True).status_int == 404

    def test_non_existent_permission(self):
        self.auth('basic1', 'basic1')
        assert self.delete('/permission_groups/user_member/1/permissions/123', {}, expect_errors=True).status_int == 404

    def test_valid(self):
        self.auth('basic1', 'basic1')
        assert self._edit_permission_groups in UserMemberPermissionGroup.get(
            self._basic_user_member_permission_group.id).permissions
        self.delete('/permission_groups/user_member/%i/permissions/%i' % (self._basic_user_member_permission_group.id,
                                                                          self._edit_permission_groups.id), {})
        assert self._edit_permission_groups not in UserMemberPermissionGroup.get(
            self._basic_user_member_permission_group.id).permissions


if __name__ == "__main__":
    unittest.main()
