import logging
import os

import sys
from renki.core.context import settings
from renki.core.lib.auth.authentication_modules.basic import BasicAuthenticationModule
from renki.core.lib.auth.db import UserToMember, User, Member, UserPermissionGroup, Permission, \
    UserMemberPermissionGroup, MemberPermissionGroup
from renki.core.lib.auth.permissions import get_permissions
from renki.core.lib.database.basic_tables import Service, Server
from sqlalchemy.orm.exc import NoResultFound

# from renki.modules.unix_user.core.unix_user_data_table import RenkiUnixUserDataTable
# from renki.modules.unix_user.core.unix_user_tables import UnixUser

logger = logging.getLogger("command_utils")


def get_service_by_id(service_id):
    """
    Get service by id
    :param service_id: service id
    :return: Service object if found, otherwise None
    """
    try:
        return Service().query.filter(Service.id == service_id).one()
    except NoResultFound:
        return


def get_users():
    """
    Get list of all users
    :return: List of users
    """
    return User().query.all()


def get_user(name):
    """
    Get user with given name.
    :param name: User name
    :return: User object if found, otherwise None
    """
    try:
        return User().query.filter(User.name == name).one()
    except NoResultFound:
        pass


def get_servers():
    """
    Get list of all servers
    :return:
    """

    return Server().query.all()


def get_server(name):
    """
    Get server
    :param name: Server name
    :return:
    """

    try:
        return Server().query.filter(Server.name == name).one()
    except NoResultFound:
        return


def get_auth_modules():
    """
    Get dictionary of authentication modules
    :return: Dictionary containing imported authentication modules
    """
    return settings.AUTH_MODULES


def add_member(member_id):
    member = Member()
    member.id = member_id
    member.save()
    return member


def get_or_create_member(member_id):
    if not Member().query.filter(Member.id == member_id).all():
        return add_member(member_id)
    else:
        return Member().query.filter(Member.id == member_id).one()


def get_member(member_id):
    try:
        return Member().query.filter(Member.id == member_id).one()
    except NoResultFound:
        return


def get_members():
    return Member().query.all()


def add_user(username, password):
    logger.info("Registering user %s with password %s" % (username, password))
    return BasicAuthenticationModule.register_user(username, password, None)


def get_or_create_user(username, password):
    if not User().query.filter(User.name == username).all():
        return add_user(username, password)
    else:
        return User().query.filter(User.name == username).one()


def add_global_permission(permission):
    perm = Permission()
    perm.name = permission.name
    perm.description = permission.description
    perm.is_global = True
    perm.save()
    return perm


def get_or_create_global_permission(permission):
    if not Permission().query.filter(Permission.name == permission.name).all():
        return add_global_permission(permission)
    else:
        return Permission().query.filter(Permission.name == permission.name).one()


def get_global_permission(permission):
    try:
        return Permission().query.filter(Permission.name == permission.name, Permission.service_id.is_(None)).one()
    except NoResultFound:
        pass


def add_service_permission(permission, description, service):
    perm = Permission()
    perm.name = permission.name
    perm.description = description
    perm.is_global = False
    perm.service_id = service.id
    perm.save()
    return perm


def get_or_create_service_permission(permission, description, service):
    if not Permission.query.filter(Permission.name == permission.name, Permission.service_id == service.id).all():
        return add_service_permission(permission, description, service)
    else:
        return Permission.query.filter(Permission.name == permission.name, Permission.service_id == service.id).one()


def get_service_permission(permission, service):
    service_permission = Permission.query.filter(Permission.name == permission.name,
                                                 Permission.service_id == service.id).all()
    if len(service_permission) != 1:
        return
    return service_permission[0]


def add_user_permission_group(group_name, description=None):
    group = UserPermissionGroup()
    group.name = group_name
    group.description = description if description else group_name
    group.save()
    return group


def get_or_create_user_permission_group(group_name):
    if not UserPermissionGroup().query.filter(UserPermissionGroup.name == group_name).all():
        return add_user_permission_group(group_name)
    return UserPermissionGroup().query.filter(UserPermissionGroup.name == group_name).one()


def get_user_permission_group(group_name):
    """
    Get user permission group matching group_name
    :param group_name: Name of group
    :return: UserPermissionGroup object if found, otherwise None
    """
    try:
        return UserPermissionGroup().query.filter(UserPermissionGroup.name == group_name).one()
    except NoResultFound:
        pass


def get_user_permission_groups():
    """
    Get all user permission groups
    :return: List of permission groups
    """
    return UserPermissionGroup().query.all()


def add_user_member_group(group_name, description=None):
    group = UserMemberPermissionGroup()
    group.name = group_name
    group.description = description if description else group_name
    group.save()
    return group


def get_or_create_user_member_group(group_name):
    if not UserMemberPermissionGroup().query.filter(UserMemberPermissionGroup.name == group_name).all():
        return add_user_member_group(group_name)
    return UserMemberPermissionGroup().query.filter(UserMemberPermissionGroup.name == group_name).one()


def get_user_member_group(group_name):
    try:
        return UserMemberPermissionGroup().query.filter(UserMemberPermissionGroup.name == group_name).one()
    except NoResultFound:
        return


def get_user_member_groups():
    """
    Get user member permission groups
    :return: List of all permission groups
    """
    return UserMemberPermissionGroup().query.all()


def add_member_permission_group(group_name, description=None):
    group = MemberPermissionGroup()
    group.name = group_name
    group.description = description if description else group_name
    group.save()
    return group


def get_or_create_member_permission_group(group_name):
    if not MemberPermissionGroup().query.filter(MemberPermissionGroup.name == group_name).all():
        return add_member_permission_group(group_name)
    return MemberPermissionGroup().query.filter(MemberPermissionGroup.name == group_name).one()


def get_member_permission_group(group_name):
    try:
        return MemberPermissionGroup().query.filter(MemberPermissionGroup.name == group_name).one()
    except NoResultFound:
        return


def get_member_permission_groups():
    return MemberPermissionGroup().query.all()


def get_or_create_user_member(user, member):
    if not UserToMember.query.filter(User.id == user.id, Member.id == member.id).all():
        return UserToMember.add_user_to_member(user=user, member=member)
    else:
        return UserToMember.query.filter(User.id == user.id, Member.id == member.id).one()


def get_user_member(user_member_id):
    """
    Get user member mapping object by id
    :param user_member_id: User member id
    :return:
    """
    user_member_id = int(user_member_id)
    try:
        return UserToMember.query.filter(UserToMember.id == user_member_id).one()
    except NoResultFound:
        pass


"""
def get_or_create_unix_user(name, member):
    if not UnixUser.query.filter(UnixUser.name == name).all():
        unix_user = UnixUser()
        unix_user.name = name
        unix_user.member = member
        unix_user.save()
        return unix_user
    return UnixUser.query.filter(UnixUser.name == name).one()"""


def sync_global_permissions():
    """
    Synchronize global permissions from code to database.
    :return:
    """
    for permission_name, permission in get_permissions().items():
        if not permission.is_global():
            continue
        perm = get_or_create_global_permission(permission)
        perm.save()


def get_or_create_server(server_name):
    """
    Find server with matching name or create new one if server isn't found
    :param server_name: name of the server (f
    :return: server identified by server_name
    """
    if not Server.query.filter(Server.name == server_name).all():
        server = Server()
        server.name = server_name
        server.save()
        return server

    return Server.query.filter(Server.name == server_name).one()


def get_or_create_service(service_name, service_type, service_servers=None):
    """
    Find service with matching name or create new one if service isn't found
    :param service_name: name of the service (for example. Lakka_port)
    :param service_type: type of the service (for example. port)
    :param service_servers: servers that handle tasks related to the service (for example. [hilla, lakka])
    :return: service identified by service_name
    """
    if not Service.query.filter(Service.name == service_name).all():
        service = Service()
        service.name = service_name
        service.service_type = service_type
        service.servers = service_servers
        service.save()
        return service

    return Service.query.filter(Service.name == service_name).one()


def get_service(service_name):
    """
    Find service with matching name
    :param service_name: service name
    :return: service identified by service_name or None if not found
    """
    service = Service.query.filter(Service.name == service_name).all()
    if service:
        return service[0]


def get_services():
    """
    Get all services
    :return: List of services
    """
    return Service.query.all()

"""
def create_limits():
    ""\"
    Crate limits for UserDataTables
    ""\"
    for table in tables.TABLES:
        if issubclass(table, RenkiUnixUserDataTable):
            limit = DefaultLimits()
            limit.table = table.__tablename__
            limit.soft_limit = table.soft_limit
            limit.hard_limit = table.hard_limit
            limit.save()"""
