import hashlib
import os
from renki.core.context import settings
from renki.core.lib.enums import JsonStatus
from flask import jsonify, Response
from json import dumps
import logging

logger = logging.getLogger('utils')

OK_STATUS = 'OK'
ERROR_STATUS = 'ERROR'
STATUS_CODES = [OK_STATUS, ERROR_STATUS]


def ok(data=None):
    """
    Creates uniform return values for bottle routes

    @param data: Data fields
    """
    if data is None:
        data = {}

    return Response(response=dumps(data), status=200, mimetype='application/json')


def error(error_message, data=None):
    """
    Creates HTTPResponse for error

    @param error_message: Error message
    @type error_message: string
    @param data: Other data
    @type data: dict
    """
    if data is None:
        data = {}

    data['status'] = JsonStatus.ERROR
    data['error'] = error_message
    return jsonify(data), 400


def unauthorized(error_message="Unauthorized", data=None):
    """
    Creates uniform return values when user is not authenticated

    @param error_message: Error message
    @type error_message: string
    @param data: Other data
    @type data: dict
    """
    if data is None:
        data = {}

    data['status'] = JsonStatus.UNAUTHORIZED
    data['error'] = error_message
    return jsonify(data), 401


def forbidden(error_message="Forbidden", data=None):
    """
    Creates uniform return values when request is denied due to invalid permissions

    @param error_message: Error message
    @type error_message: string
    @param data: Other data
    @type data: dict
    """
    if data is None:
        data = {}

    data['status'] = JsonStatus.FORBIDDEN
    data['error'] = error_message
    return jsonify(data), 403


def not_found(error_message="Requested page not found", data=None):
    """
    Creates uniform return values when requested page is not found

    @param error_message: Error message
    @type error_message: string
    @param data: Other data
    @type data: dict
    """
    if data is None:
        data = {}

    data['status'] = JsonStatus.NOT_FOUND
    data['error'] = error_message
    return jsonify(data), 404


def not_allowed(error_message="Requested method not allowed", data=None):
    """
    Creates uniform return values when HTTP method is not allowed

    @param error_message: Error message
    @type error_message: string
    @param data: Other data
    @type data: dict
    """
    if data is None:
        data = {}

    data['status'] = JsonStatus.NOT_ALLOWED
    data['error'] = error_message
    return jsonify(data), 405


def conflict(error_message="Conflict", data=None):
    """
    Creates uniform return values when requested method is not allowed

    @param error_message: Error message
    @type error_message: string
    @param data: Other data
    @type data: dict
    """
    if data is None:
        data = {}

    data['status'] = JsonStatus.CONFLICT
    data['error'] = error_message
    return jsonify(data), 409


def server_error(error_message='Internal server error', data=None):
    """
    Creates uniform return values when requested caused internal server error

    @param error_message: Error message
    @type error_message: string
    @param data: Other data
    @type data: dict
    """
    if data is None:
        data = {}

    data['status'] = JsonStatus.SERVER_ERROR
    data['error'] = error_message
    return jsonify(data), 500


def generate_token(size=40):
    """
    Generate random token
    """
    return hashlib.sha512(settings.SECRET.encode("utf-8") + os.urandom(128)).hexdigest()[:size]
