from renki.core.lib.test_utils import BasicTest
from renki.core.lib.auth.authentication_modules.basic import BasicAuthenticationModule
from renki.core.lib.auth.db import Permission, UserPermissionGroup
from renki.core.lib.database.table import db
import unittest
from renki.core.lib.auth.basic_permissions import ViewServers, EditServers
import jsonschema

__unittest = True


class TestServersListRoute(BasicTest):
    def setUp(self):
        super(TestServersListRoute, self).setUp()

        self._lakka = self.get_or_create_server('Lakka')
        db.session.flush()

        self._lakka_port = self.get_or_create_service('Lakka_port', 'port', [self._lakka])
        db.session.commit()

        self._view_servers = Permission()
        self._view_servers.name = ViewServers.name
        self._view_servers.is_global = True
        self._view_servers.save()

        self._edit_servers = Permission()
        self._edit_servers.name = EditServers.name
        self._edit_servers.is_global = True
        self._edit_servers.save()

        db.session.commit()

        self._basic_user_1 = BasicAuthenticationModule.register_user('basic1', 'basic1')
        self._basic_user_2 = BasicAuthenticationModule.register_user('basic2', 'basic2')
        db.session.commit()

        self._admin_permission_group = UserPermissionGroup()
        self._admin_permission_group.name = 'Admin'
        self._admin_permission_group.description = 'Admins'
        self._admin_permission_group.add_permission(self._edit_servers)
        self._admin_permission_group.add_permission(self._view_servers)
        self._admin_permission_group.add_user(self._basic_user_1)
        db.session.commit()

    def tearDown(self):
        super(TestServersListRoute, self).tearDown()

    def test_not_authenticated(self):
        assert self.get('/servers', {}, expect_errors=True).status_int == 401

    def test_no_permissions(self):
        self.auth('basic2', 'basic2')

        assert self.get('/servers', {}, expect_errors=True).status_int == 403

    def test_valid_single(self):
        self.auth('basic1', 'basic1')

        data = self.get('/servers').json

        assert len(data) == 1
        assert data[0]['name'] == 'Lakka'

    def test_valid_multiple(self):
        self.auth('basic1', 'basic1')

        hilla = self.get_or_create_server('Hilla')
        hilla.save()
        db.session.commit()

        data = self.get('/servers').json

        assert len(data) == 2
        assert data[0]['name'] == 'Lakka'
        assert data[1]['name'] == 'Hilla'


class TestServersInfoRoute(BasicTest):
    _server_response_schema = {
        'type': 'object',
        'properties': {
            'id': {'type': 'integer'},
            'name': {'type': 'string'},
            'services': {
                'type': 'array',
                'items': {
                    'type': 'object',
                    'properties': {
                        'id': {'type': 'integer'},
                        'name': {'type': 'string'},
                        'type': {'type': 'string'}
                    }
                }
            }
        },
        'required': ['id', 'name', 'services'],
        'additionalProperties': False
    }

    def setUp(self):
        super(TestServersInfoRoute, self).setUp()

        self._lakka = self.get_or_create_server('Lakka')
        db.session.flush()

        self._lakka_port = self.get_or_create_service('Lakka_port', 'port', [self._lakka])
        db.session.commit()

        self._view_servers = Permission()
        self._view_servers.name = ViewServers.name
        self._view_servers.is_global = True
        self._view_servers.save()

        self._edit_servers = Permission()
        self._edit_servers.name = EditServers.name
        self._edit_servers.is_global = True
        self._edit_servers.save()

        db.session.commit()

        self._basic_user_1 = BasicAuthenticationModule.register_user('basic1', 'basic1')
        self._basic_user_2 = BasicAuthenticationModule.register_user('basic2', 'basic2')
        db.session.commit()

        self._admin_permission_group = UserPermissionGroup()
        self._admin_permission_group.name = 'Admin'
        self._admin_permission_group.description = 'Admins'
        self._admin_permission_group.add_permission(self._edit_servers)
        self._admin_permission_group.add_permission(self._view_servers)
        self._admin_permission_group.add_user(self._basic_user_1)
        db.session.commit()

    def tearDown(self):
        super(TestServersInfoRoute, self).tearDown()

    def test_not_authenticated(self):
        assert self.get('/servers/1', {}, expect_errors=True).status_int == 401

    def test_no_permissions(self):
        self.auth('basic2', 'basic2')

        assert self.get('/servers/1', {}, expect_errors=True).status_int == 403

    def test_valid_request(self):
        self.auth('basic1', 'basic1')

        data = self.get('/servers/1').json
        jsonschema.validate(data, self._server_response_schema)

        assert data['id'] == self._lakka.id
        assert data['name'] == self._lakka.name

    def test_invalid_request(self):
        self.auth('basic1', 'basic1')

        assert self.get('/servers/server', {}, expect_errors=True).status_int == 404

if __name__ == "__main__":
    unittest.main()
