from renki.core.lib.test_utils import BasicTest
from renki.core.lib.auth.permissions import GlobalPermission, ServicePermission
from renki.core.lib.auth.authentication_modules.basic import BasicAuthenticationModule
from renki.core.lib.auth.db import Permission, Member, UserPermissionGroup, AuthTokens, MemberPermissionGroup, \
    UserMemberPermissionGroup, UserToMember
from renki.core.lib.database.table import db
import unittest
import jsonschema

__unittest = True

ModifyInfoPermission = GlobalPermission('session_test_modify_info', 'Session test modify info')
CreatePortPermission = ServicePermission('session_test_create_port', 'Session test create port')
NonExistingGlobalPermission = GlobalPermission('session_test_non_existing_global_perm',
                                               'Session test non existing global permission')
NonExistingServicePermission = ServicePermission('session_test_non_existing_service_perm',
                                                 'Session test non existing service permission')


class TestSessionInfoRoute(BasicTest):
    def setUp(self):
        super(TestSessionInfoRoute, self).setUp()

        self._basic_user_1 = BasicAuthenticationModule.register_user('basic1', 'basic1')

        self._basic_member_1 = Member()
        db.session.add(self._basic_member_1)

        db.session.commit()

    def tearDown(self):
        super(TestSessionInfoRoute, self).tearDown()

    def test_not_authenticated(self):
        assert self.get('/session/info', {}, expect_errors=True).status_int == 401

    def test_no_member(self):
        self.auth('basic1', 'basic1')

        data = self.get('/session/info').json
        assert 'member_id' not in data

    def test_member_selected(self):
        UserToMember.add_user_to_member(self._basic_user_1, self._basic_member_1)
        db.session.commit()

        auth_token = self.auth('basic1', 'basic1')

        data = self.get('/session/info', {'auth_token': auth_token}).json
        assert 'memberId' in data
        assert data['memberId'] == self._basic_member_1.id


class TestSessionListMembersRoute(BasicTest):
    _members_schema = {
        'type': 'array',
        'items': {
            'type': 'integer'
        }
    }

    def setUp(self):
        super(TestSessionListMembersRoute, self).setUp()

        self._basic_user_1 = BasicAuthenticationModule.register_user('basic1', 'basic1')

        self._basic_member_1 = Member()
        db.session.add(self._basic_member_1)

        self._basic_member_2 = Member()
        db.session.add(self._basic_member_2)

        db.session.commit()

    def test_not_authenticated(self):
        assert self.get('/session/members', {}, expect_errors=True).status_int == 401

    def test_no_member(self):
        self.auth('basic1', 'basic1')

        data = self.get('/session/members').json
        jsonschema.validate(data, self._members_schema)

        assert len(data) == 0

    def test_single_member(self):
        UserToMember.add_user_to_member(self._basic_user_1, self._basic_member_1)
        db.session.commit()

        self.auth('basic1', 'basic1')

        data = self.get('/session/members').json
        jsonschema.validate(data, self._members_schema)

        assert len(data) == 1
        assert self._basic_member_1.id in data

    def test_multiple_members(self):
        UserToMember.add_user_to_member(self._basic_user_1, self._basic_member_1)
        UserToMember.add_user_to_member(self._basic_user_1, self._basic_member_2)
        db.session.commit()

        self.auth('basic1', 'basic1')

        data = self.get('/session/members').json
        jsonschema.validate(data, self._members_schema)

        assert len(data) == 2
        assert self._basic_member_1.id in data
        assert self._basic_member_2.id in data


class TestSessionSelectMemberRoute(BasicTest):
    def setUp(self):
        super(TestSessionSelectMemberRoute, self).setUp()

        self._basic_user_1 = BasicAuthenticationModule.register_user('basic1', 'basic1')
        self._basic_member_1 = Member()
        db.session.add(self._basic_member_1)
        self._basic_member_2 = Member()
        db.session.add(self._basic_member_2)

        db.session.commit()

    def test_not_authenticated(self):
        assert self.post('/session/members', {'id': 123}, expect_errors=True).status_int == 401

    def test_no_members(self):
        self.auth('basic1', 'basic1')

        assert self.post('/session/members', {'id': 123}, expect_errors=True).status_int == 403

    def test_invalid_member(self):
        UserToMember.add_user_to_member(self._basic_user_1, self._basic_member_1)
        db.session.commit()

        self.auth('basic1', 'basic1')

        assert self.post('/session/members', {'id': self._basic_member_2.id}, expect_errors=True) \
                   .status_int == 403

    def test_single_member(self):
        auth_token = self.auth('basic1', 'basic1')

        UserToMember.add_user_to_member(self._basic_user_1, self._basic_member_1)
        db.session.commit()

        assert AuthTokens.get_token(auth_token).member_id is None

        data = self.post('/session/members', {'id': self._basic_member_1.id}).json
        assert AuthTokens.get_token(auth_token).member_id == self._basic_member_1.id

    def test_multiple_members(self):
        auth_token = self.auth('basic1', 'basic1')

        UserToMember.add_user_to_member(self._basic_user_1, self._basic_member_1)
        UserToMember.add_user_to_member(self._basic_user_1, self._basic_member_2)
        db.session.commit()

        assert AuthTokens.get_token(auth_token).member_id is None

        self.post('/session/members?authToken=%s' % auth_token, {'id': self._basic_member_1.id})
        assert AuthTokens.get_token(auth_token).member_id == self._basic_member_1.id

        self.post('/session/members?authToken=%s' % auth_token, {'id': self._basic_member_2.id})
        assert AuthTokens.get_token(auth_token).member_id == self._basic_member_2.id


class TestSessionListPermissionsRoute(BasicTest):
    _session_permissions_schema = {
        'type': 'object',
        'properties': {
            'permissions': {
                'type': 'array',
                'items': {
                    'type': 'object',
                    'properties': {
                        'id': {
                            'type': 'integer'
                        },
                        'name': {
                            'type': 'string'
                        },
                        'serviceId': {
                            'type': 'integer'
                        },
                        'isGlobal': {
                            'type': 'boolean'
                        }
                    },
                    'required': ['id', 'name', 'serviceId', 'isGlobal'],
                    'additionalProperties': False
                }
            }
        }
    }

    def setUp(self):
        super(TestSessionListPermissionsRoute, self).setUp()

        self._lakka = self.get_or_create_server('Lakka')
        db.session.flush()

        self._lakka_port = self.get_or_create_service('Lakka_port', 'port', [self._lakka])
        db.session.commit()

        self._modify_info = Permission()
        self._modify_info.name = ModifyInfoPermission.name
        self._modify_info.is_global = True
        db.session.add(self._modify_info)

        self._create_lakka_port = Permission()
        self._create_lakka_port.name = CreatePortPermission.name
        self._create_lakka_port.service_id = self._lakka_port.id
        self._create_lakka_port.is_global = False
        db.session.add(self._create_lakka_port)
        db.session.commit()

        self._basic_user_1 = BasicAuthenticationModule.register_user('basic1', 'basic1')
        self._basic_user_2 = BasicAuthenticationModule.register_user('basic2', 'basic2')

        self._basic_member_1 = Member()
        db.session.add(self._basic_member_1)

        self._basic_member_2 = Member()
        db.session.add(self._basic_member_2)
        db.session.commit()

        self._basic_user_member_1 = UserToMember.add_user_to_member(self._basic_user_1, self._basic_member_1)
        self._basic_user_member_2 = UserToMember.add_user_to_member(self._basic_user_2, self._basic_member_2)
        db.session.commit()

        self._basic_user_permission_group = UserPermissionGroup()
        self._basic_user_permission_group.name = 'basic_user'
        self._basic_user_permission_group.users.append(self._basic_user_1)
        db.session.add(self._basic_user_permission_group)

        self._basic_member_permission_group = MemberPermissionGroup()
        self._basic_member_permission_group.name = 'basic_member'
        self._basic_member_permission_group.members.append(self._basic_member_1)
        self._basic_member_permission_group.members.append(self._basic_member_2)
        db.session.add(self._basic_member_permission_group)

        self._basic_user_member_permission_group = UserMemberPermissionGroup()
        self._basic_user_member_permission_group.name = 'basic_user_member'
        # self._basic_user_member_permission_group.add_user_member(self._basic_user_member_1)
        # self._basic_user_member_permission_group.add_user_member(self._basic_user_member_2)
        db.session.add(self._basic_user_member_permission_group)

        db.session.commit()

        self._basic_user_token_1 = BasicAuthenticationModule.authenticate('basic1', 'basic1')
        self._basic_user_identity_1 = AuthTokens.get_token(self._basic_user_token_1['authToken'])

        self._basic_user_token_2 = BasicAuthenticationModule.authenticate('basic2', 'basic2')
        self._basic_user_identity_2 = AuthTokens.get_token(self._basic_user_token_2['authToken'])

    def test_not_authenticated(self):
        assert self.get('/session/permissions', {}, expect_errors=True).status_int == 401

    def test_user_permission(self):
        """
        Test that permissions associated with user work as expected
        """
        self.auth('basic1', 'basic1')

        data = self.get('/session/permissions').json

        assert 'modulePermissions' in data and not data['modulePermissions']

        self._basic_user_permission_group.permissions.append(self._create_lakka_port)
        db.session.add(self._basic_user_permission_group)
        db.session.commit()

        data = self.get('/session/permissions').json
        jsonschema.validate(data, self._session_permissions_schema)

        permissions = data['modulePermissions']
        assert (len(permissions) == 1)

        create_port = permissions[0]
        assert create_port['id'] == self._create_lakka_port.id
        assert create_port['name'] == self._create_lakka_port.name
        assert create_port['serviceId'] == self._create_lakka_port.service_id

    def test_single_member(self):
        """
        Test that permissions of selected member identity work
        """
        self.auth('basic1', 'basic1')

        data = self.get('/session/permissions').json
        assert 'modulePermissions' in data and not data['modulePermissions']

        self._basic_member_permission_group.permissions.append(self._create_lakka_port)
        db.session.add(self._basic_member_permission_group)
        db.session.commit()

        data = self.get('/session/permissions').json
        assert 'modulePermissions' in data and not data['modulePermissions']

        self._basic_user_member_permission_group.add_user_member(self._basic_user_member_1)
        self._basic_user_member_permission_group.permissions.append(self._create_lakka_port)
        db.session.add(self._basic_user_member_permission_group)
        db.session.commit()

        data = self.get('/session/permissions').json
        jsonschema.validate(data, self._session_permissions_schema)

        permissions = data['modulePermissions']

        assert (len(permissions) == 1)
        assert (permissions[0]['id'] == self._create_lakka_port.id)
        assert (permissions[0]['name'] == self._create_lakka_port.name)
        assert (permissions[0]['serviceId'] == self._create_lakka_port.service_id)


if __name__ == "__main__":
    unittest.main()
