import logging

from renki.core.lib.utils import generate_token
from renki.core.lib.auth.authentication import AuthenticationModule
from renki.core.lib.auth.db import BasicAuthUser
from renki.core.lib.exceptions import AuthenticationFailed, AlreadyExist
from renki.core.lib.database.table import db
from sqlalchemy.orm.exc import NoResultFound


logger = logging.getLogger('BasicAuthentication')


class BasicAuthenticationModule(AuthenticationModule):
    """
    BasicAuthenticationModule provides a basic authentication against local database
    """
    NAME = 'BASIC'

    @classmethod
    def _register_user(cls, username: str, password: str, params: dict):
        if BasicAuthUser.query.filter(BasicAuthUser.name == username).count() != 0:
            raise AlreadyExist('User %s already exists' % username)

        user = BasicAuthUser()
        user.name = username
        user.set_password(password)
        db.session.add(user)

    @classmethod
    def _authenticate(cls, username: str, password: str):
        """
        Authenticate user using username and password
        :return: token associated with session
        """
        try:
            user = BasicAuthUser.query.filter(BasicAuthUser.name == username).one()

            if user.check_password(password) is True:
                token = generate_token()
                return {'user': user, 'token': token, 'expires': None}
        except NoResultFound as e:
            logger.exception(e)
            raise AuthenticationFailed('Invalid username or password')

        raise AuthenticationFailed('Invalid username or password')

    @classmethod
    def _validate_token(cls, key: str):
        """
        Checks if token is valid.
        For basic module this is always true as token has been generated and put straight to AuthTokens
        """
        return True

    @classmethod
    def require_password(cls):
        return True

    def __repr__(self):
        return self.NAME
