from renki.core.lib.renkiapi import renkiapi
from renki.core.lib.exceptions import DoesNotExist
from renki.core.lib.auth.db import Permission
from renki.core.lib.utils import ok as ret_ok, conflict as ret_conflict, not_found as ret_not_found,\
    forbidden as ret_denied
from renki.core.lib.auth.basic_permissions import ViewPermissions, EditPermissions
from renki.core.lib.database.table import db
from flask import Blueprint

blueprint = Blueprint('permissions', __name__)


@renkiapi('GET', '/permissions', blueprint=blueprint)
def permissions_list(identity):
    """
    List all available permissions
    :returns permissions in following format
    permissions: {
        id: {
            name: name of permission,
            description: description of permission,
            service_id: permission_service id (if there is one),
            is_global: flag telling if permission is global or not
        }
    }
    """

    if not identity.has_permission(ViewPermissions):
        return ret_denied()

    permissions = {}

    for permission in Permission.query.all():
        permissions[permission.id] = {
            'name': permission.name,
            'description': permission.description,
            'serviceId': permission.service_id,
            'isGlobal': permission.is_global
        }

    return ret_ok({'permissions': permissions})


@renkiapi('POST', '/permissions',
          json={
              'type': 'object',
              'properties': {
                  'name': {'type': 'string'},
                  'description': {'type': 'string'},
                  'serviceId': {'type': 'integer'},
                  'isGlobal': {'type': 'integer'}
              }
          },
          response={
              'type': 'object',
              'properties': {
                  'permission': {'type': 'integer'}
              }
          }, blueprint=blueprint)
def permissions_add(identity, data):
    """
    List all available permissions
    :returns id of created permission
    permission: {
        id: 123
    }
    """
    if not identity.has_permission(EditPermissions):
        return ret_denied()

    name = data['name']
    description = data['description']
    service_id = data['serviceId']
    is_global = data['isGlobal']

    if is_global:
        if Permission.query.filter(Permission.name == name, Permission.service_id.is_(None)).count():
            return ret_conflict('Permission with name %s and service_id %i already exists' % (name, service_id))
    else:
        if Permission.query.filter(Permission.name == name, Permission.service_id == service_id).count():
            return ret_conflict('Permission with name %s and service_id %i already exists' % (name, service_id))

    permission = Permission()
    permission.name = name
    permission.description = description
    if is_global:
        permission.is_global = True
        permission.service_id = None
    else:
        permission.is_global = False
        permission.service_id = service_id
    permission.save()

    db.session.commit()

    return ret_ok({'permission': permission.id})


@renkiapi('DELETE', '/permissions/<int:id>',
          url_params={
              'type': 'object',
              'properties': {
                  'id': {'type': 'integer'}
              },
              'required': ['id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permissions_delete(identity, data):
    """
    Delete permission with id 'id'
    """
    if not identity.has_permission(EditPermissions):
        return ret_denied()

    permission_id = int(data['id'])

    try:
        permission = Permission.get(permission_id)
    except DoesNotExist:
        return ret_not_found('Permission with id %i not found' % permission_id)

    db.session.delete(permission)
    db.session.commit()

    return ret_ok()


@renkiapi('PUT', '/permissions/<int:id>',
          url_params={
              'type': 'object',
              'properties': {
                  'id': {'type': 'integer'}
              },
              'required': ['id'],
              'additionalProperties': False
          },
          json={
              'type': 'object',
              'properties': {
                  'name': {'type': 'string'},
                  'description': {'type': 'string'},
                  'serviceId': {'type': 'integer'},
                  'isGlobal': {'type': 'boolean'}
              },
              'required': ['name', 'description', 'serviceId', 'isGlobal'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permissions_update_permission(identity, data):
    """
    Update permission with id 'permission_id' to match values posted as json
    """
    if not identity.has_permission(EditPermissions):
        return ret_denied()

    name = data['name']
    description = data['description']
    service_id = int(data['serviceId'])
    is_global = bool(data['isGlobal'])
    permission_id = int(data['id'])

    try:
        permission = Permission.get(permission_id)
    except DoesNotExist:
        return ret_not_found('Permission with id %i not found' % permission_id)

    permission.name = name
    permission.description = description
    permission.is_global = is_global
    if not permission.is_global:
        permission.service_id = service_id
    permission.save()

    db.session.commit()

    return ret_ok()
