from renki.core.lib.test_utils import BasicTest
from renki.core.lib.auth.permissions import GlobalPermission, ServicePermission
from renki.core.lib.auth.db import Permission, Member, UserPermissionGroup, AuthTokens, MemberPermissionGroup,\
    UserMemberPermissionGroup, UserToMember
from renki.core.lib.auth.authentication_modules.basic import BasicAuthenticationModule
from renki.core.lib.database.table import db
import unittest

__unittest = True

ModifyInfoPermission = GlobalPermission('permission_test_modify_info', 'Test modify info')
CreatePortPermission = ServicePermission('permission_test_create_port', 'Test create port')
NonExistingGlobalPermission = GlobalPermission('permission_test_non_existing_global_perm',
                                               'Test non existing global permission')
NonExistingServicePermission = ServicePermission('permission_test_non_existing_service_perm',
                                                 'Test non existing service permission')


class TestPermissions(BasicTest):
    def setUp(self):
        super(TestPermissions, self).setUp()

        self._lakka = self.get_or_create_server('Lakka')
        self._hilla = self.get_or_create_server('Hilla')
        db.session.flush()

        self._hilla_port = self.get_or_create_service('Hilla_port', 'port', [self._hilla])
        self._lakka_port = self.get_or_create_service('Lakka_port', 'port', [self._lakka])
        db.session.commit()

        self._modify_info = Permission()
        self._modify_info.name = ModifyInfoPermission.name
        self._modify_info.is_global = True
        self._modify_info.save()

        self._create_hilla_port = Permission()
        self._create_hilla_port.name = CreatePortPermission.name
        self._create_hilla_port.service_id = self._hilla_port.id
        self._create_hilla_port.is_global = False
        self._create_hilla_port.save()

        self._create_lakka_port = Permission()
        self._create_lakka_port.name = CreatePortPermission.name
        self._create_lakka_port.service_id = self._lakka_port.id
        self._create_lakka_port.is_global = False
        self._create_lakka_port.save()
        db.session.commit()

        self._basic_user_1 = BasicAuthenticationModule.register_user('basic1', 'basic1')
        self._basic_user_2 = BasicAuthenticationModule.register_user('basic2', 'basic2')

        self._basic_member_1 = Member()
        self._basic_member_1.save()

        self._basic_member_2 = Member()
        self._basic_member_2.save()
        db.session.commit()

        self._basic_user_member_1 = UserToMember.add_user_to_member(self._basic_user_1, self._basic_member_1)
        self._basic_user_member_2 = UserToMember.add_user_to_member(self._basic_user_2, self._basic_member_2)
        db.session.commit()

        self._basic_user_permission_group = UserPermissionGroup()
        self._basic_user_permission_group.name = 'basic_user'
        self._basic_user_permission_group.users.append(self._basic_user_1)
        self._basic_user_permission_group.users.append(self._basic_user_2)
        self._basic_user_permission_group.save()

        self._basic_member_permission_group = MemberPermissionGroup()
        self._basic_member_permission_group.name = 'basic_member'
        self._basic_member_permission_group.members.append(self._basic_member_1)
        self._basic_member_permission_group.members.append(self._basic_member_2)
        self._basic_member_permission_group.save()

        self._basic_user_member_permission_group = UserMemberPermissionGroup()
        self._basic_user_member_permission_group.name = 'basic_user_member'
        self._basic_user_member_permission_group.add_user_member(self._basic_user_member_1)
        self._basic_user_member_permission_group.add_user_member(self._basic_user_member_2)
        self._basic_user_member_permission_group.save()

        db.session.commit()

        self._basic_user_token_1 = BasicAuthenticationModule.authenticate('basic1', 'basic1')
        self._basic_user_identity_1 = AuthTokens.get_token(self._basic_user_token_1['authToken'])

        self._basic_user_token_2 = BasicAuthenticationModule.authenticate('basic2', 'basic2')
        self._basic_user_identity_2 = AuthTokens.get_token(self._basic_user_token_2['authToken'])

    def test_has_permission_global_user(self):
        assert(not self._basic_user_identity_1.has_permission(ModifyInfoPermission))

        self._basic_user_permission_group.permissions.append(self._modify_info)
        self._basic_user_permission_group.save()
        db.session.commit()

        assert(self._basic_user_identity_1.has_permission(ModifyInfoPermission))
        assert(not self._basic_user_identity_1.has_permission(CreatePortPermission))
        assert(not self._basic_user_identity_1.has_permission(NonExistingGlobalPermission))

    def test_has_permission_service_user(self):
        assert(not self._basic_user_identity_1.has_permission(CreatePortPermission, self._hilla_port.id))

        self._basic_user_permission_group.permissions.append(self._create_hilla_port)

        assert(self._basic_user_identity_1.has_permission(CreatePortPermission, self._hilla_port.id))
        assert(not self._basic_user_identity_1.has_permission(CreatePortPermission, -1))
        assert(not self._basic_user_identity_1.has_permission(NonExistingServicePermission, 123))

    def test_has_permission_global_member(self):
        assert(not self._basic_user_identity_1.has_permission(ModifyInfoPermission))

        # Add permission for member to modify_info but don't add it to user_member_mapping yet
        self._basic_member_permission_group.add_permission(self._modify_info)
        assert(not self._basic_user_identity_1.has_permission(ModifyInfoPermission))

        # Add permission also to user_member_permissions and test that user is now permitted to do it
        self._basic_user_member_permission_group.add_permission(self._modify_info)
        assert(self._basic_user_identity_1.has_permission(ModifyInfoPermission))

    def test_has_permission_service_member(self):
        assert(not self._basic_user_identity_1.has_permission(CreatePortPermission, self._hilla_port.id))

        # Add permission for member to modify_info but don't add it to user_member_mapping yet
        self._basic_member_permission_group.add_permission(self._create_hilla_port)
        assert(not self._basic_user_identity_1.has_permission(CreatePortPermission, self._hilla_port.id))

        # Add permission also to user_member_permissions and test that user is now permitted to do it
        self._basic_user_member_permission_group.add_permission(self._create_hilla_port)
        assert(self._basic_user_identity_1.has_permission(CreatePortPermission, self._hilla_port.id))
        assert(not self._basic_user_identity_1.has_permission(CreatePortPermission, self._lakka_port.id))

if __name__ == "__main__":
    unittest.main()
