import json
import yaml
import importlib
import os
import logging
from sqlalchemy import Table
from sqlalchemy.exc import IntegrityError, ProgrammingError
from renki.core.context import settings
from renki import core as renki_core
from renki.core.lib.exceptions import RenkiException

logger = logging.getLogger('migrations')


class FixtureError(RenkiException):
    pass


class FixtureFile:
    def __init__(self, module_name: str, fixture_class: str, fixture_file: str, absolute_path: str):
        self.module_name = module_name
        self.fixture_class = fixture_class
        self.fixture_file = fixture_file
        self.absolute_path = absolute_path

        self.anchors = set()
        self.aliases = set()

        self.dependencies = set()
        self.dependants = set()

        self._load_anchors_and_aliases()

    def _load_anchors_and_aliases(self):
        """
        Load anchors and aliases used in YAML -file
        """

        with open(self.absolute_path) as file:
            try:
                # Scan yaml stream to get scanning tokens to determine anchors and aliases used in the file
                scan = yaml.scan(file, yaml.SafeLoader)
                for item in scan:
                    if isinstance(item, yaml.AnchorToken):
                        self.anchors.add(item.value)
                    elif isinstance(item, yaml.AliasToken):
                        self.aliases.add(item.value)
            except yaml.YAMLError as e:
                logger.exception('Failed to load file: %s', self.absolute_path, exc_info=e)
                raise


def apply_fixtures(db, module_name=None, directory_filter=None):
    all_fixtures = find_all_fixtures()

    fixtures_to_apply = []
    for fixture in all_fixtures:
        if (module_name is None or fixture.module_name == module_name) and \
                (directory_filter is None or fixture.fixture_class == directory_filter):
            fixtures_to_apply.append(fixture)

    anchors_to_fixtures = {}
    for fixture in fixtures_to_apply:
        # Map anchors to fixture files providing them
        for anchor in fixture.anchors:
            # Although YAML supports redefining anchors, neither we or PyYAML do
            if anchor in anchors_to_fixtures:
                error_str = 'Anchor "{0}" defined in "{1}" redefined in "{2}"!'
                raise FixtureError(error_str.format(anchor, anchors_to_fixtures[anchor].fixture_file,
                                                    fixture.fixture_file))
            else:
                anchors_to_fixtures[anchor] = fixture

    # Find out what fixtures depend on which other fixtures
    for node in fixtures_to_apply:
        if not node.aliases.issubset(node.anchors):
            leftover_aliases = node.aliases.difference(node.anchors)
            for alias in leftover_aliases:
                if alias not in anchors_to_fixtures:
                    raise FixtureError('Anchor "{0}" aliased in "{1}" not found!'.format(alias, node.fixture_file))
                else:
                    node.dependencies.add(anchors_to_fixtures[alias])
                    anchors_to_fixtures[alias].dependants.add(node)

    # Sort fixtures by dependants to solve chains from bottom to top.
    # This isn't necessary but more a nice-to-have.
    fixtures_to_apply.sort(key=lambda x: len(x.dependants))

    # Resolve all dependency chains between fixtures
    ordered_fixtures = []
    while len(fixtures_to_apply) > 0:
        # Start from one (more or less random) node and proceed up in dependency chain. Since each fixture can be
        # applied after it's dependencies have been applied this leads to one (possibly incomplete) chain to be solved
        # and repeating it until all fixtures have been processed ensures that all fixtures get applied in (one
        # possible) valid order.
        fixture_node = fixtures_to_apply[0]

        processed = []
        queue = [fixture_node]

        # Run BFS through the dependencies
        while len(queue) > 0:
            fixture_node = queue[0]
            del queue[0]

            processed.insert(0, fixture_node)

            try:
                fixtures_to_apply.remove(fixture_node)
            except KeyError:
                # If the fixture has already been removed from fixture_nodes, we're dealing with cyclic dependencies
                error_str = 'Detected cyclic dependency while processing "{0}"'
                raise FixtureError(error_str.format(fixture_node.fixture_file))

            # Add dependencies of node to process them
            queue.extend(fixture_node.dependencies)

            # Remove dependency from dependants to consider it solved
            for node in fixture_node.dependants:
                node.dependencies.remove(fixture_node)

            fixture_node.dependants.clear()

        ordered_fixtures.extend(processed)

    fixture_files = []
    for fixture in ordered_fixtures:
        fixture_files.append(fixture.absolute_path)

    apply_fixtures_real(db, fixture_files)


def find_all_fixtures():
    """
    Find all fixtures in core and all loaded modules
    :return: list of fixtures (list of 4-tuples)
    """
    fixtures = []

    fixtures.extend(find_module_fixtures('core', renki_core))
    for loaded_module in settings.MODULES:
        fixtures.extend(find_module_fixtures(loaded_module.name, loaded_module))

    return fixtures


def split_path(path):
    """
    Splits path to list of folders
    :param path: a path to be split
    :return: list of different parts that form the path
    """
    parts = []

    while path != '':
        (path, last) = os.path.split(path)
        parts.insert(0, last)

    return parts


def find_module_fixtures(module_name, loaded_module):
    """
    Find all YAML fixtures in a module
    :param module_name: name of the module
    :param loaded_module: loaded python module used as a root for fixture search
    :return: tuple(module_name, fixture_class, fixture_file, absolute_path)
    """
    fixture_files = []
    module_path = os.path.dirname(loaded_module.__file__)
    fixture_root = os.path.abspath(module_path + "/fixtures/")

    if os.path.isdir(fixture_root):
        for fixture_dir in os.walk(fixture_root):
            for fixture_file in fixture_dir[2]:
                if fixture_file.endswith('.yml') or fixture_file.endswith('.yaml'):
                    absolute_path = os.path.abspath(fixture_dir[0] + "/" + fixture_file)
                    fixture_class = split_path(os.path.relpath(absolute_path, fixture_root))[0]
                    fixture_files.append(FixtureFile(module_name, fixture_class, fixture_file, absolute_path))
    else:
        logger.info('No fixtures found for module {0}'.format(loaded_module.__name__))

    return fixture_files


def yaml_compose_and_keep_anchors(self):
    """
    Custom compose_document function that mimics behaviour of PyYAML's composers compose_document function but
    preserves anchors instead of clearing them
    """
    self.get_event()
    node = self.compose_node(None, None)
    self.get_event()
    return node

yaml.SafeLoader.compose_document = yaml_compose_and_keep_anchors


def apply_fixtures_real(db, files: []):
    if files is None:
        return

    conn = db.engine.connect()
    metadata = db.metadata

    anchors = {}
    for filename in files:
        with open(filename) as file:
            try:
                loader = yaml.SafeLoader(file)
                loader.anchors = anchors
                while loader.check_data():
                    data = loader.get_data()

                    for fixture in data:
                        if 'model' in fixture:
                            module_name, class_name = fixture['model'].rsplit('.', 1)
                            loaded_module = importlib.import_module(module_name)
                            model = getattr(loaded_module, class_name)
                            for fields in fixture['records']:
                                obj = model(**fields)
                                db.session.merge(obj)
                                db.session.commit()
                        elif 'table' in fixture:
                            table = Table(fixture['table'], metadata)

                            for fields in fixture['records']:
                                try:
                                    conn.execute(table.insert(), fields)
                                except ProgrammingError as e:
                                    logger.exception('Failed to apply fixture file: %s', filename, exc_info=e)
                                    return
                                except IntegrityError:
                                    # Since tables are only used for m2m relationships, we simply ignore
                                    # integrity errors in these cases
                                    pass
                        else:
                            raise ValueError('Fixture missing a "model" or "table" field {0}'.
                                             format(json.dumps(fixture)))

                anchors.update(loader.anchors)
            finally:
                loader.dispose()
