from renki.core.lib.test_utils import BasicTest
from renki.core.lib.auth.authentication_modules.basic import BasicAuthenticationModule
from renki.core.lib.auth.db import Permission, UserPermissionGroup
from renki.core.lib.auth.basic_permissions import ViewUserPermissionGroups, EditUserPermissionGroups, ViewPersonalData
from renki.core.lib.database.table import db
import unittest
import jsonschema


class UserPermissionGroupTest(BasicTest):
    def setUp(self):
        super(UserPermissionGroupTest, self).setUp()

        self._view_permission_groups = Permission()
        self._view_permission_groups.name = ViewUserPermissionGroups.name
        self._view_permission_groups.is_global = True
        self._view_permission_groups.save()

        self._edit_permission_groups = Permission()
        self._edit_permission_groups.name = EditUserPermissionGroups.name
        self._edit_permission_groups.is_global = True
        self._edit_permission_groups.save()

        self._dummy_permission = Permission()
        self._dummy_permission.name = ViewPersonalData.name
        self._dummy_permission.is_global = True
        self._dummy_permission.save()

        db.session.commit()

        self._basic_user_1 = BasicAuthenticationModule.register_user('basic1', 'basic1')
        self._basic_user_2 = BasicAuthenticationModule.register_user('basic2', 'basic2')
        self._dummy_user = BasicAuthenticationModule.register_user('dummy', 'dummy')
        db.session.commit()

        self._admin_permission_group = UserPermissionGroup()
        self._admin_permission_group.name = 'Admin'
        self._admin_permission_group.description = 'Admins'
        self._admin_permission_group.add_permission(self._edit_permission_groups)
        self._admin_permission_group.add_permission(self._view_permission_groups)
        self._admin_permission_group.add_user(self._basic_user_1)
        db.session.commit()


class TestListUserPermissionGroupsRoute(UserPermissionGroupTest):
    _user_permission_groups_schema = {
        'user_permission_groups': {
            'type': 'object',
            'patternProperties': {
                '^[0-9]+$': {
                    'type': 'object',
                    'properties': {
                        'name': {
                            'type': 'string'
                        },
                        'description': {
                            'type': 'string'
                        }
                    },
                    'required': ['name', 'description'],
                    'additionalProperties': False
                }
            }
        },
        'additionalProperties': True
    }

    def setUp(self):
        super(TestListUserPermissionGroupsRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.get('/permission_groups/user', {}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')

        assert self.get('/permission_groups/user', {}, expect_errors=True).status_int == 403

    def test_valid(self):
        self.auth('basic1', 'basic1')

        data = self.get('/permission_groups/user', {}).json

        jsonschema.validate(data, self._user_permission_groups_schema)

        admin_permission_group = data['user_permission_groups'][str(self._admin_permission_group.id)]
        assert admin_permission_group['name'] == self._admin_permission_group.name
        assert admin_permission_group['description'] == self._admin_permission_group.description


class TestAddUserPermissionGroupRoute(UserPermissionGroupTest):
    def setUp(self):
        super(TestAddUserPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.post('/permission_groups/user',
                         {'name': 'test_group',
                          'description': 'test group is a test group'}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')

        assert self.post('/permission_groups/user',
                         {'name': 'test_group',
                          'description': 'test group is a test group'}, expect_errors=True).status_int == 403

    def test_duplicate(self):
        self.auth('basic1', 'basic1')

        permission_group = UserPermissionGroup()
        permission_group.name = 'test_group'
        permission_group.description = 'test group is a test group'
        permission_group.save()
        db.session.commit()

        assert self.post('/permission_groups/user',
                         {'name': 'test_group',
                          'description': 'test group is a test group'}, expect_errors=True).status_int == 409

    def test_valid(self):
        self.auth('basic1', 'basic1')

        assert UserPermissionGroup.query.filter(UserPermissionGroup.name == 'test_group').count() == 0

        self.post('/permission_groups/user', {'name': 'test_group', 'description': 'test group is a test group'})

        assert UserPermissionGroup.query.filter(UserPermissionGroup.name == 'test_group').count() == 1


class TestUpdateUserPermissionGroupRoute(UserPermissionGroupTest):
    def setUp(self):
        super(TestUpdateUserPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.put('/permission_groups/user/1', {}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')

        assert self.put('/permission_groups/user/1',
                        {'name': 'test_group',
                         'description': 'test group is a test group'}, expect_errors=True).status_int == 403

    def test_non_existent_group(self):
        self.auth('basic1', 'basic1')

        assert self.put('/permission_groups/user/123',
                        {'name': 'test_group',
                         'description': 'test group is a test group'}, expect_errors=True).status_int == 404

    def test_valid(self):
        self.auth('basic1', 'basic1')

        assert UserPermissionGroup.get(self._admin_permission_group.id).name == self._admin_permission_group.name

        self.put('/permission_groups/user/%i' % self._admin_permission_group.id,
                 {'name': 'test_group', 'description': 'test group is a test group'})

        assert UserPermissionGroup.get(self._admin_permission_group.id).name == 'test_group'


class TestDeleteUserPermissionGroupRoute(UserPermissionGroupTest):
    def setUp(self):
        super(TestDeleteUserPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.delete('/permission_groups/user/1', {}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')
        assert self.delete('/permission_groups/user/1', {}, expect_errors=True).status_int == 403

    def test_non_existent_group(self):
        self.auth('basic1', 'basic1')
        assert self.delete('/permission_groups/user/123', {}, expect_errors=True).status_int == 404

    def test_valid(self):
        self.auth('basic1', 'basic1')
        assert UserPermissionGroup.query.filter(UserPermissionGroup.id == self._admin_permission_group.id).count()
        self.delete('/permission_groups/user/%i' % self._admin_permission_group.id, {})
        assert not UserPermissionGroup.query.filter(UserPermissionGroup.id == self._admin_permission_group.id).count()


class TestListUsersInUserPermissionGroupRoute(UserPermissionGroupTest):
    _users_schema = {
        'users': {
            'type': 'object',
            'patternProperties': {
                '^[0-9]+$': {
                    'type': 'object',
                    'properties': {
                        'name': {
                            'type': 'string'
                        },
                        'auth_module': {
                            'type': 'string'
                        }
                    },
                    'required': ['name', 'auth_module'],
                    'additionalProperties': False
                }
            }
        },
        'additionalProperties': True
    }

    def setUp(self):
        super(TestListUsersInUserPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.get('/permission_groups/user/1/users', {}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')
        assert self.get('/permission_groups/user/1/users', {}, expect_errors=True).status_int == 403

    def test_non_existent_group(self):
        self.auth('basic1', 'basic1')
        assert self.get('/permission_groups/user/123/users', {}, expect_errors=True).status_int == 404

    def test_valid(self):
        self.auth('basic1', 'basic1')
        data = self.get('/permission_groups/user/%i/users' % self._admin_permission_group.id, {}).json
        jsonschema.validate(data, self._users_schema)
        user = data['users'][str(self._basic_user_1.id)]
        assert user['name'] == self._basic_user_1.name


class TestAddUserToUserPermissionGroupRoute(UserPermissionGroupTest):
    def setUp(self):
        super(TestAddUserToUserPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.post('/permission_groups/user/1/users',
                         {'user_id': self._dummy_user.id}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')
        assert self.post('/permission_groups/user/1/users',
                         {'user_id': self._dummy_user.id}, expect_errors=True).status_int == 403

    def test_non_existent_group(self):
        self.auth('basic1', 'basic1')
        assert self.post('/permission_groups/user/123/users',
                         {'user_id': self._dummy_user.id}, expect_errors=True).status_int == 404

    def test_non_existent_user(self):
        self.auth('basic1', 'basic1')
        assert self.post('/permission_groups/user/1/users',
                         {'user_id': 123}, expect_errors=True).status_int == 404

    def test_valid(self):
        self.auth('basic1', 'basic1')
        assert self._basic_user_2 not in self._admin_permission_group.users
        self.post('/permission_groups/user/%i/users' % self._admin_permission_group.id,
                  {'user_id': self._basic_user_2.id})
        assert self._basic_user_2 in self._admin_permission_group.users


class TestDeleteUserFromUserPermissionGroupRoute(UserPermissionGroupTest):
    def setUp(self):
        super(TestDeleteUserFromUserPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.delete('/permission_groups/user/1/users/123', {}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')
        assert self.delete('/permission_groups/user/1/users/123', {}, expect_errors=True).status_int == 403

    def test_non_existent_group(self):
        self.auth('basic1', 'basic1')
        assert self.delete('/permission_groups/user/123/users/%i' % self._dummy_user.id, {},
                           expect_errors=True).status_int == 404

    def test_non_existent_user(self):
        self.auth('basic1', 'basic1')
        assert self.delete('/permission_groups/user/1/users/123', {}, expect_errors=True).status_int == 404

    def test_valid(self):
        self.auth('basic1', 'basic1')
        assert self._basic_user_1 in UserPermissionGroup.get(self._admin_permission_group.id).users
        self.delete('/permission_groups/user/%i/users/%i' % (self._admin_permission_group.id, self._basic_user_1.id),
                    {})
        assert self._basic_user_1 not in UserPermissionGroup.get(self._admin_permission_group.id).users


class TestListPermissionsInUserPermissionGroupRoute(UserPermissionGroupTest):
    _permissions_schema = {
        'permissions': {
            'type': 'object',
            'patternProperties': {
                '^[0-9]+$': {
                    'type': 'object',
                    'properties': {
                        'name': {
                            'type': 'string'
                        },
                        'description': {
                            'type': 'string'
                        },
                        'service_id': {
                            'type': 'integer'
                        },
                        'is_global': {
                            'type': 'boolean'
                        }
                    },
                    'required': ['name', 'description', 'service_id', 'is_global'],
                    'additionalProperties': False
                }
            }
        },
        'additionalProperties': True
    }

    def setUp(self):
        super(TestListPermissionsInUserPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.get('/permission_groups/user/1/permissions', {}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')
        assert self.get('/permission_groups/user/1/permissions', {}, expect_errors=True).status_int == 403

    def test_non_existent_group(self):
        self.auth('basic1', 'basic1')
        assert self.get('/permission_groups/user/123/permissions', {}, expect_errors=True).status_int == 404

    def test_valid(self):
        self.auth('basic1', 'basic1')
        data = self.get('/permission_groups/user/%i/permissions' % self._admin_permission_group.id, {}).json
        jsonschema.validate(data, self._permissions_schema)
        permission = data['permissions'][str(self._edit_permission_groups.id)]
        assert permission['name'] == self._edit_permission_groups.name


class TestAddPermissionToUserPermissionGroupRoute(UserPermissionGroupTest):
    def setUp(self):
        super(TestAddPermissionToUserPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.post('/permission_groups/user/1/permissions',
                         {'permission_id': self._dummy_permission.id}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')
        assert self.post('/permission_groups/user/1/permissions',
                         {'permission_id': self._dummy_permission.id}, expect_errors=True).status_int == 403

    def test_non_existent_group(self):
        self.auth('basic1', 'basic1')
        assert self.post('/permission_groups/user/123/permissions',
                         {'permission_id': self._dummy_permission.id}, expect_errors=True).status_int == 404

    def test_non_existent_permission(self):
        self.auth('basic1', 'basic1')
        assert self.post('/permission_groups/user/1/permissions',
                         {'permission_id': 123}, {}, expect_errors=True).status_int == 404

    def test_valid(self):
        self.auth('basic1', 'basic1')
        assert self._dummy_permission not in self._admin_permission_group.permissions
        self.post('/permission_groups/user/%i/permissions' % self._admin_permission_group.id,
                  {'permission_id': self._dummy_permission.id})
        assert self._dummy_permission in self._admin_permission_group.permissions


class TestDeletePermissionFromUserPermissionGroupRoute(UserPermissionGroupTest):
    def setUp(self):
        super(TestDeletePermissionFromUserPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.delete('/permission_groups/user/1/permissions/123', {}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')
        assert self.delete('/permission_groups/user/1/permissions/123', {}, expect_errors=True).status_int == 403

    def test_non_existent_group(self):
        self.auth('basic1', 'basic1')
        assert self.delete('/permission_groups/user/123/permissions/1', {}, expect_errors=True).status_int == 404

    def test_non_existent_permission(self):
        self.auth('basic1', 'basic1')
        assert self.delete('/permission_groups/user/1/permissions/123', {}, expect_errors=True).status_int == 404

    def test_valid(self):
        self.auth('basic1', 'basic1')
        assert self._edit_permission_groups in UserPermissionGroup.get(self._admin_permission_group.id).permissions
        self.delete('/permission_groups/user/%i/permissions/%i' % (self._admin_permission_group.id,
                                                                   self._edit_permission_groups.id), {})
        assert self._edit_permission_groups not in UserPermissionGroup.get(self._admin_permission_group.id).permissions


if __name__ == "__main__":
    unittest.main()
