from renki.core.lib.test_utils import BasicTest
from renki.core.lib.auth.authentication_modules.basic import BasicAuthenticationModule
from renki.core.lib.auth.db import Permission, UserPermissionGroup, MemberPermissionGroup, Member
from renki.core.lib.auth.basic_permissions import ViewMemberPermissionGroups, EditMemberPermissionGroups,\
    ViewPersonalData
from renki.core.lib.database.table import db
import unittest
import jsonschema


class MemberPermissionGroupTest(BasicTest):
    def setUp(self):
        super(MemberPermissionGroupTest, self).setUp()

        self._view_permission_groups = Permission()
        self._view_permission_groups.name = ViewMemberPermissionGroups.name
        self._view_permission_groups.is_global = True
        self._view_permission_groups.save()

        self._edit_permission_groups = Permission()
        self._edit_permission_groups.name = EditMemberPermissionGroups.name
        self._edit_permission_groups.is_global = True
        self._edit_permission_groups.save()

        self._dummy_permission = Permission()
        self._dummy_permission.name = ViewPersonalData.name
        self._dummy_permission.is_global = True
        self._dummy_permission.save()

        db.session.commit()

        self._basic_user_1 = BasicAuthenticationModule.register_user('basic1', 'basic1')
        self._basic_user_2 = BasicAuthenticationModule.register_user('basic2', 'basic2')

        self._basic_member_1 = Member()
        self._basic_member_1.save()

        self._dummy_member = Member()
        self._dummy_member.save()

        db.session.commit()

        self._admin_permission_group = UserPermissionGroup()
        self._admin_permission_group.name = 'Admin'
        self._admin_permission_group.description = 'Admins'
        self._admin_permission_group.add_permission(self._edit_permission_groups)
        self._admin_permission_group.add_permission(self._view_permission_groups)
        self._admin_permission_group.add_user(self._basic_user_1)

        self._basic_member_permission_group = MemberPermissionGroup()
        self._basic_member_permission_group.name = 'Basic'
        self._basic_member_permission_group.description = 'Basic members'
        self._basic_member_permission_group.add_permission(self._view_permission_groups)
        self._basic_member_permission_group.add_permission(self._edit_permission_groups)
        self._basic_member_permission_group.add_member(self._basic_member_1)
        db.session.commit()


class TestListMemberPermissionGroupsRoute(MemberPermissionGroupTest):
    _member_permission_groups_schema = {
        'member_permission_groups': {
            'type': 'object',
            'patternProperties': {
                '^[0-9]+$': {
                    'type': 'object',
                    'properties': {
                        'name': {
                            'type': 'string'
                        },
                        'description': {
                            'type': 'string'
                        }
                    },
                    'required': ['name', 'description'],
                    'additionalProperties': False
                }
            }
        },
        'additionalProperties': True
    }

    def setUp(self):
        super(TestListMemberPermissionGroupsRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.get('/permission_groups/member', {}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')

        assert self.get('/permission_groups/member', {}, expect_errors=True).status_int == 403

    def test_valid(self):
        self.auth('basic1', 'basic1')

        data = self.get('/permission_groups/member', {}).json

        jsonschema.validate(data, self._member_permission_groups_schema)

        basic_member_permission_group = data[0]
        assert basic_member_permission_group['name'] == self._basic_member_permission_group.name
        assert basic_member_permission_group['description'] == self._basic_member_permission_group.description


class TestAddMemberPermissionGroupRoute(MemberPermissionGroupTest):
    def setUp(self):
        super(TestAddMemberPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.post('/permission_groups/member',
                         {'name': 'test_group',
                          'description': 'test group is a test group'}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')

        assert self.post('/permission_groups/member',
                         {'name': 'test_group',
                          'description': 'test group is a test group'}, expect_errors=True).status_int == 403

    def test_duplicate(self):
        self.auth('basic1', 'basic1')

        permission_group = MemberPermissionGroup()
        permission_group.name = 'test_group'
        permission_group.description = 'test group is a test group'
        permission_group.save()
        db.session.commit()

        assert self.post('/permission_groups/member',
                         {'name': 'test_group',
                          'description': 'test group is a test group'}, expect_errors=True).status_int == 409

    def test_valid(self):
        self.auth('basic1', 'basic1')

        assert MemberPermissionGroup.query.filter(MemberPermissionGroup.name == 'test_group').count() == 0

        self.post('/permission_groups/member', {'name': 'test_group', 'description': 'test group is a test group'})

        assert MemberPermissionGroup.query.filter(MemberPermissionGroup.name == 'test_group').count() == 1


class TestUpdateMemberPermissionGroupRoute(MemberPermissionGroupTest):
    def setUp(self):
        super(TestUpdateMemberPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.put('/permission_groups/member/1', {}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')

        assert self.put('/permission_groups/member/1',
                        {'name': 'test_group',
                         'description': 'test group is a test group'}, expect_errors=True).status_int == 403

    def test_non_existent_group(self):
        self.auth('basic1', 'basic1')

        assert self.put('/permission_groups/member/123',
                        {'name': 'test_group',
                         'description': 'test group is a test group'}, expect_errors=True).status_int == 404

    def test_valid(self):
        self.auth('basic1', 'basic1')

        assert MemberPermissionGroup.get(
            self._basic_member_permission_group.id).name == self._basic_member_permission_group.name

        self.put('/permission_groups/member/%i' % self._basic_member_permission_group.id,
                 {'name': 'test_group', 'description': 'test group is a test group'})

        assert MemberPermissionGroup.get(self._basic_member_permission_group.id).name == 'test_group'


class TestDeleteMemberPermissionGroupRoute(MemberPermissionGroupTest):
    def setUp(self):
        super(TestDeleteMemberPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.delete('/permission_groups/member/1', {}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')
        assert self.delete('/permission_groups/member/1', {}, expect_errors=True).status_int == 403

    def test_non_existent_group(self):
        self.auth('basic1', 'basic1')
        assert self.delete('/permission_groups/member/123', {}, expect_errors=True).status_int == 404

    def test_valid(self):
        self.auth('basic1', 'basic1')
        assert MemberPermissionGroup.query.filter(MemberPermissionGroup.id == self._admin_permission_group.id).count()
        self.delete('/permission_groups/member/%i' % self._admin_permission_group.id, {})
        assert not MemberPermissionGroup.query.\
            filter(MemberPermissionGroup.id == self._admin_permission_group.id).count()


class TestListMembersInMemberPermissionGroupRoute(MemberPermissionGroupTest):
    _members_schema = {
        'members': {
            'type': 'array',
            'items': {
                'type': 'integer'
            }
        },
        'additionalProperties': True
    }

    def setUp(self):
        super(TestListMembersInMemberPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.get('/permission_groups/member/1/members', {}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')
        assert self.get('/permission_groups/member/1/members', {}, expect_errors=True).status_int == 403

    def test_non_existent_group(self):
        self.auth('basic1', 'basic1')
        assert self.get('/permission_groups/member/123/members', {}, expect_errors=True).status_int == 404

    def test_valid(self):
        self.auth('basic1', 'basic1')
        data = self.get('/permission_groups/member/%i/members' % self._admin_permission_group.id, {}).json
        jsonschema.validate(data, self._members_schema)


class TestAddMemberToMemberPermissionGroupRoute(MemberPermissionGroupTest):
    def setUp(self):
        super(TestAddMemberToMemberPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.post('/permission_groups/member/1/members',
                         {'member_id': self._dummy_member.id}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')
        assert self.post('/permission_groups/member/1/members',
                         {'member_id': self._dummy_member.id}, expect_errors=True).status_int == 403

    def test_non_existent_group(self):
        self.auth('basic1', 'basic1')
        assert self.post('/permission_groups/member/123/members',
                         {'member_id': self._dummy_member.id}, expect_errors=True).status_int == 404

    def test_non_existent_member(self):
        self.auth('basic1', 'basic1')
        assert self.post('/permission_groups/member/1/members',
                         {'member_id': 123}, expect_errors=True).status_int == 404

    def test_valid(self):
        self.auth('basic1', 'basic1')
        assert self._dummy_member not in self._basic_member_permission_group.members
        self.post('/permission_groups/member/%i/members' % self._basic_member_permission_group.id,
                  {'member_id': self._dummy_member.id})
        assert self._dummy_member in self._basic_member_permission_group.members


class TestDeleteMemberFromMemberPermissionGroupRoute(MemberPermissionGroupTest):
    def setUp(self):
        super(TestDeleteMemberFromMemberPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.delete('/permission_groups/member/1/members/123', {}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')
        assert self.delete('/permission_groups/member/1/members/123', {}, expect_errors=True).status_int == 403

    def test_non_existent_group(self):
        self.auth('basic1', 'basic1')
        assert self.delete('/permission_groups/member/123/members/%i' % self._dummy_member.id, {},
                           expect_errors=True).status_int == 404

    def test_non_existent_member(self):
        self.auth('basic1', 'basic1')
        assert self.delete('/permission_groups/member/1/members/123', {}, expect_errors=True).status_int == 404

    def test_valid(self):
        self.auth('basic1', 'basic1')
        assert self._basic_member_1 in MemberPermissionGroup.get(self._basic_member_permission_group.id).members
        self.delete('/permission_groups/member/%i/members/%i' %
                    (self._basic_member_permission_group.id, self._basic_member_1.id), {})
        assert self._basic_member_1 not in MemberPermissionGroup.get(self._basic_member_permission_group.id).members


class TestListPermissionsInMemberPermissionGroupRoute(MemberPermissionGroupTest):
    _permissions_schema = {
        'permissions': {
            'type': 'object',
            'patternProperties': {
                '^[0-9]+$': {
                    'type': 'object',
                    'properties': {
                        'name': {
                            'type': 'string'
                        },
                        'description': {
                            'type': 'string'
                        },
                        'service_id': {
                            'type': 'integer'
                        },
                        'is_global': {
                            'type': 'boolean'
                        }
                    },
                    'required': ['name', 'description', 'service_id', 'is_global'],
                    'additionalProperties': False
                }
            }
        },
        'additionalProperties': True
    }

    def setUp(self):
        super(TestListPermissionsInMemberPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.get('/permission_groups/member/1/permissions', {}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')
        assert self.get('/permission_groups/member/1/permissions', {}, expect_errors=True).status_int == 403

    def test_non_existent_group(self):
        self.auth('basic1', 'basic1')
        assert self.get('/permission_groups/member/123/permissions', {}, expect_errors=True).status_int == 404

    def test_valid(self):
        self.auth('basic1', 'basic1')
        data = self.get('/permission_groups/member/%i/permissions' % self._admin_permission_group.id, {}).json
        jsonschema.validate(data, self._permissions_schema)
        permission = data['permissions'][str(self._edit_permission_groups.id)]
        assert permission['name'] == self._edit_permission_groups.name


class TestAddPermissionToMemberPermissionGroupRoute(MemberPermissionGroupTest):
    def setUp(self):
        super(TestAddPermissionToMemberPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.post('/permission_groups/member/1/permissions',
                         {'permission_id': self._dummy_permission.id}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')
        assert self.post('/permission_groups/member/1/permissions',
                         {'permission_id': self._dummy_permission.id}, expect_errors=True).status_int == 403

    def test_non_existent_group(self):
        self.auth('basic1', 'basic1')
        assert self.post('/permission_groups/member/123/permissions',
                         {'permission_id': self._dummy_permission.id}, expect_errors=True).status_int == 404

    def test_non_existent_permission(self):
        self.auth('basic1', 'basic1')
        assert self.post('/permission_groups/member/1/permissions',
                         {'permission_id': 123}, {}, expect_errors=True).status_int == 404

    def test_valid(self):
        self.auth('basic1', 'basic1')
        assert self._dummy_permission not in self._basic_member_permission_group.permissions
        self.post('/permission_groups/member/%i/permissions' % self._admin_permission_group.id,
                  {'permission_id': self._dummy_permission.id})
        assert self._dummy_permission in self._basic_member_permission_group.permissions


class TestDeletePermissionFromMemberPermissionGroupRoute(MemberPermissionGroupTest):
    def setUp(self):
        super(TestDeletePermissionFromMemberPermissionGroupRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.delete('/permission_groups/member/1/permissions/123', {}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        self.auth('basic2', 'basic2')
        assert self.delete('/permission_groups/member/1/permissions/123', {}, expect_errors=True).status_int == 403

    def test_non_existent_group(self):
        self.auth('basic1', 'basic1')
        assert self.delete('/permission_groups/member/123/permissions/1', {}, expect_errors=True).status_int == 404

    def test_non_existent_permission(self):
        self.auth('basic1', 'basic1')
        assert self.delete('/permission_groups/member/1/permissions/123', {}, expect_errors=True).status_int == 404

    def test_valid(self):
        self.auth('basic1', 'basic1')
        assert self._edit_permission_groups in MemberPermissionGroup.get(self._admin_permission_group.id).permissions
        self.delete('/permission_groups/member/%i/permissions/%i' % (self._admin_permission_group.id,
                                                                     self._edit_permission_groups.id), {})
        assert self._edit_permission_groups not in MemberPermissionGroup.get(
            self._admin_permission_group.id).permissions


if __name__ == "__main__":
    unittest.main()
