from sqlalchemy.orm import relationship
from sqlalchemy import Column, Integer, Boolean, ForeignKey, Unicode
from sqlalchemy.orm.exc import NoResultFound
from sqlalchemy.ext.declarative import declared_attr

from renki.core.lib.history_meta import Versioned
from renki.core.lib.database.table import RenkiDataTable
from renki.core.lib.communication.ticket_tables import TicketGroup
from renki.core.lib.exceptions import Invalid, SoftLimitReached, HardLimitReached
from renki.core.lib.auth.db import DefaultLimits
from renki.modules.unix_user.core.unix_user_tables import UnixUserLimits, UnixUser
from renki.modules.unix_user.core.unix_user_methods import create_ticket_unix_user, create_delete_ticket_unix_user


class RenkiUnixUserDataTable(RenkiDataTable, Versioned):
    __abstract__ = True

    @declared_attr
    def unix_user_id(cls):
        return Column('unix_user_id', Integer, ForeignKey('unix_users.id'), nullable=False)

    @declared_attr
    def unix_user(cls):
        return relationship(UnixUser)

    @declared_attr
    def type(cls):
        return Column('type', Unicode)

    waiting = Column('waiting', Boolean, nullable=False, default=False)
    rejected = Column('rejected', Boolean, nullable=False, default=False)

    soft_limit = None
    hard_limit = None

    @declared_attr
    def ticket_group_id(cls):
        return Column(Integer, ForeignKey('ticket_groups.id'))

    @declared_attr
    def ticket_group(cls):
        return relationship(TicketGroup)

    def real_save(self, commit=False):
        return RenkiDataTable.save(self, commit)

    def save(self, commit=False):
        if not self.waiting:
            create_ticket_unix_user(self)
        else:
            return RenkiDataTable.save(self, commit)

    def delete(self):
        """
        Delete this object from database
        """
        create_delete_ticket_unix_user(self)
        return RenkiDataTable.delete(self)

    # Needs to be overwritten for tables that don't have service_id
    def get_service_id(self):
        if 'service_id' in self.__table__.columns:
            return self.service_id

        raise Invalid('No service_id')

    @classmethod
    def get_ticket_type(cls):
        raise Invalid('Ticket type not defined')

    @classmethod
    def get_limits_for_user(cls, unix_user_id):
        limits = {'soft_limit': cls.soft_limit, 'hard_limit': cls.hard_limit}

        try:
            default_limit = DefaultLimits.query.filter(DefaultLimits.table == cls.__tablename__).one()
            limits = {'soft_limit': default_limit.soft_limit, 'hard_limit': default_limit.hard_limit}
        except NoResultFound:
            pass

        try:
            limit = UnixUserLimits.query.filter(UnixUserLimits.table == cls.__tablename__,
                                                UnixUserLimits.unix_user_id == unix_user_id).one()
            limits = {'soft_limit': limit.soft_limit, 'hard_limit': limit.hard_limit}
        except NoResultFound:
            pass

        return limits

    @classmethod
    def count_unix_user_entries(cls, unix_user_id):
        return cls.query.filter(cls.unix_user_id == unix_user_id).count()

    @classmethod
    def validate_add(cls, unix_user_id):
        limits = cls.get_limits_for_user(unix_user_id)
        entries = cls.count_unix_user_entries(unix_user_id)
        if entries >= limits['hard_limit']:
            raise HardLimitReached("Hard limit for ports reached")
        elif entries >= limits['soft_limit']:
            raise SoftLimitReached("Soft limit for ports reached")
