from sqlalchemy.orm.exc import NoResultFound, MultipleResultsFound

from renki.core.lib.auth.db import AuthTokens
from renki.core.lib.exceptions import NoUnixUserFound, MultipleUnixUsersFound, NoMemberSelected
from renki.core.lib.communication.ticket_tables import TicketGroup
from renki.core.lib.database.basic_tables import Service
from renki.core.lib.database.table import db
from renki.modules.unix_user.core.unix_user_tables import UnixUser, UnixUserTicket

def extend_auth_token():
    """
    Adds method get_unix_user to AuthToken-class
    """
    def get_unix_user(self):
        """
        Get unix user owned by member
        :param self: Instance of Member class
        :return: Unix user owned by member
        """
        if self.member is None:
            raise NoMemberSelected('No member has been selected to act as')

        try:
            unix_user = UnixUser.query.filter(UnixUser.member_id == self.member_id).one()

            return unix_user
        except NoResultFound:
            raise NoUnixUserFound('Member %i does not have any unix users' % self.member_id)
        except MultipleResultsFound:
            raise MultipleUnixUsersFound('Member %i has multiple unix users' % self.member_id)

    AuthTokens.get_unix_user = get_unix_user

extend_auth_token()


def create_ticket_unix_user(user_data_table):
    ticket_group = TicketGroup()
    ticket_group.save(commit=False)

    user_data_table.ticket_group = ticket_group
    user_data_table.real_save(commit=False)
    db.session.flush()

    unix_user = UnixUser.get(user_data_table.unix_user_id)

    service_id = user_data_table.get_service_id()
    ticket_type = user_data_table.get_ticket_type()

    new_data = user_data_table.as_dict()

    history = user_data_table.__history_mapper__.class_

    old_data = history.query.filter(history.id == user_data_table.id).order_by(history.version.desc()).first()
    if old_data:
        tmp = old_data.to_dict()
        old_data = {k: tmp[k] for k in user_data_table.as_dict().keys()}

    service = Service.query.filter(Service.id == service_id).one()

    for server in service.servers:
        ticket = UnixUserTicket()
        ticket.old_data = old_data
        ticket.new_data = new_data
        ticket.ticket_group = ticket_group
        ticket.server = server
        ticket.ticket_module = ticket_type
        if unix_user is not None:
            ticket.unix_user = unix_user
        ticket.save(commit=False)


def create_delete_ticket_unix_user(user_data_table):
    ticket_group = TicketGroup()
    ticket_group.save()
    db.session.flush()

    service_id = user_data_table.get_service_id()
    ticket_type = user_data_table.get_ticket_type()

    unix_user = UnixUser.get(user_data_table.unix_user_id)

    service = Service.query.filter(Service.id == service_id).one()
    for server in service.servers:
        ticket = UnixUserTicket()
        ticket.old_data = user_data_table.as_dict()
        ticket.new_data = None
        ticket.ticket_group_id = ticket_group.id
        ticket.server = server
        ticket.ticket_module = ticket_type
        if unix_user is not None:
            ticket.unix_user = unix_user
        ticket.save()
