"""
This settings module is dynamically populated by lib.check_settings.
"""

import os


class SettingError(Exception):
    pass


class Condition(object):
    def __init__(self, name):
        self.name = name

    def __or__(self, other):
        c = self.check

        def f(*args, **kwargs):
            return c(*args, **kwargs) or other.check(*args, **kwargs)

        self.check = f
        return self

    def __and__(self, other):
        c = self.check

        def f(*args, **kwargs):
            return c(*args, **kwargs) and other.check(*args, **kwargs)

        self.check = f
        return self


class Required(Condition):
    """
    This setting doesn't have default value
    """

    def __init__(self, name):
        super(Required, self).__init__(name)
        self.name = name

    def check(self, settings, value=None):
        if not value:
            raise SettingError("Value for setting %s is required" % self.name)
        return value


class RequiredIf(Condition):
    def __init__(self, name, condition):
        super(RequiredIf, self).__init__(name)
        self.name = name
        self.condition = condition

    def check(self, settings, value=None):
        if not self.condition(settings):
            raise SettingError("Invalid value %s for setting %s" % (value, self.name))
        return value


class CheckValue(Condition):
    def __init__(self, name, condition):
        super().__init__(name)
        self.name = name
        self.condition = condition

    def check(self, settings, value):
        if not self.condition(settings):
            raise SettingError("Invalid value %s for setting %s" % (value, self.name))
        return value


class IsFile(Condition):
    def __init__(self, name):
        super().__init__(name)
        self.name = name

    def check(self, settings, value):
        p = os.path.abspath(os.path.expanduser(value))
        if not os.path.isfile(p):
            raise SettingError("File %s does not exist, invalid value for setting %s" % (p, self.name))
        return p


LOGGING = {
    'version': 1,
    'disable_existing_loggers': True,
    'formatters': {
        'verbose': {
            'format': '%(levelname)s %(asctime)s %(module)s %(process)d %(thread)d %(message)s'
        },
        'simple': {
            '()': 'logging.Formatter',
            'format': '%(asctime)-20s %(levelname)s %(module)s %(message)s'
        }
    },
    'filters': {
    },
    'handlers': {
        'null': {
            'level': 'DEBUG',
            'class': 'logging.NullHandler',
        },
        'console': {
            'class': 'logging.StreamHandler',
            'formatter': 'verbose'
        }
    },
    'loggers': {
        'server': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'INFO',
        },
        'admin': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'INFO',
        },
        'utils': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'ticket': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'INFO',
        },
        'module_database': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'module_dns_zone': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'module_domain': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'module_port': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'module_repository': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'module_dummy': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'tickets_done': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
        'RenkiSocket': {
            'handlers': ['console'],
            'propagate': True,
            'level': 'DEBUG',
        },
    }
}

DEBUG = True

RENKI_SRV_NAME = Required('RENKI_SRV_NAME')

RENKI_ADDRESS = Required('RENKI_ADDRESS')

RENKI_LOGIN_NAME = Required('RENKI_LOGIN_NAME')
RENKI_LOGIN_PASSWORD = Required('RENKI_LOGIN_PASSWORD')