#!/usr/bin/env python
import sys
from renki.srv.context import settings
from renki.srv.lib.handler import task_manager
from renki.common.communication.client import RenkiClient
from time import sleep
from renki.common.exceptions import RenkiException, NotAuthenticated, AuthenticationFailed
import logging


def main():
    try:
        settings.from_cli('srv')
        logger = logging.getLogger('server')

        logger.info("Starting server")

        r = RenkiClient(settings.CORE_HOST)

        auth_failure_threshold = 9
        auth_failure_sleep = 5
        auth_failure_counter = 0
        auth_success = False
        while not auth_success:
            try:
                r.auth(settings.USERNAME, settings.PASSWORD)
                auth_success = True
            except AuthenticationFailed:
                if auth_failure_counter < auth_failure_threshold:
                    auth_failure_counter += 1
                    logger.info("Attempt %d to authenticate to RenkiCore failed. %d attempts remaining",
                                auth_failure_counter, auth_failure_threshold - auth_failure_counter + 1)
                    sleep(auth_failure_sleep)
                else:
                    logger.fatal('Unable to start RenkiSrv: Failed to authenticate to RenkiCore.')
                    return

        logger.debug("Authenticated to RenkiCore")

        task_manager.start_all()

        while True:
            logger.info('Polling tasks from core')
            try:
                data = r.get('tickets/%s' % settings.NAME)
            except NotAuthenticated:
                logger.info('Server returned not authenticated. Trying to authenticate again')
                r.auth(settings.USERNAME, settings.PASSWORD)
                continue
            except Exception as e:
                logger.exception("Failed to parse")
                continue

            print(data)

            # TODO : validate format of data
            for ticket in data['tickets']:
                task_type = ticket['ticket_module']
                task_manager.dispatch_task(task_type, task=ticket)
            sleep(data['next_poll'])
    except KeyboardInterrupt:
        pass
    finally:
        logger.info('Stopping RenkiSrv')
        task_manager.stop_all()
        logger.info('Waiting for task handlers to stop')
        task_manager.join_all()
        logger.info('Server stopped')

if __name__ == '__main__':
    try:
        main()
    except KeyboardInterrupt:
        sys.exit(0)
