from renki.srv.context import settings
from renki.srv.lib.exceptions import UnknownTaskType
from renki.common.communication.client import RenkiClient
import threading
import queue
import logging


logger = logging.getLogger('server')


class TaskHandler(threading.Thread):
    def __init__(self):
        threading.Thread.__init__(self)
        self._tasks = queue.Queue()
        self._stopped = False

    def push_task(self, task):
        self._tasks.put(task)

    def pop_task(self):
        return self._tasks.get()

    def start(self):
        super(TaskHandler, self).start()

    def stop(self):
        self._stopped = True

    def run(self):
        while True:
            if self._stopped:
                return

            try:
                task = self._tasks.get(True, 1)
                try:
                    self.process(task)
                    self.report_success(task, 'Task finished successfully')
                except Exception as e:
                    self.report_error(task, str(e))
                self._tasks.task_done()
            except queue.Empty:
                pass

    def process(self, task):
        pass

    def report_error(self, task, error):
        r = RenkiClient(settings.CORE_HOST)
        r.auth(settings.USERNAME, settings.PASSWORD)

        ticket_id = task['id']

        logger.debug('Ticket %i failed with error %s' % (ticket_id, error))

        r.post('/tickets/%i' % ticket_id, {'error': error})

    def report_success(self, task, comment):
        r = RenkiClient(settings.CORE_HOST)
        r.auth(settings.USERNAME, settings.PASSWORD)

        ticket_id = task['id']

        logger.debug('Ticket %i succeeded with comment %s' % (ticket_id, comment))

        r.post('/tickets/%i' % ticket_id, {'comment': comment})


class TaskManager(object):
    def __init__(self):
        self.task_handlers = {}

    def register_task_handler(self, task_type, task_handler):
        logger.info("Registering %s task handler" % task_type)
        self.task_handlers[task_type] = task_handler()

    def dispatch_task(self, task_type, task):
        if task_type not in self.task_handlers:
            logger.error("Got unknown task type %s" % task_type)
            raise UnknownTaskType("Unknown task type %s" % task_type)
        self.task_handlers[task_type].push_task(task)

    def start_all(self):
        for handler in self.task_handlers.values():
            handler.start()

    def stop_all(self):
        for handler in self.task_handlers.values():
            handler.stop()

    def join_all(self):
        for handler in self.task_handlers.values():
            handler.join()

task_manager = TaskManager()
